#include "rdmlib.h"
#include "string.h"

#ifdef RTTLOGGINGSET
#include "loggerservice.h"
#endif

#ifdef RTTLOGGINGSET
// LOGGER ENABLE ------------
#define LOGENABLE 0
#endif

uint16_t GetRdmMessageLength(uint8_t dataLength)
{
	return dataLength + RDMHEADERLENGTH + 2;
}

void EncodeRdmMessage(rdm_message_header_t msg, uint8_t *data, uint8_t *buf)
{
	buf[0]  = (SCRDM);                                 // SC_RDM
	buf[1]  = (SCSUBMESSAGE);                          // SC_SUB_MESSAGE
	buf[2]  = ((uint8_t)(RDMHEADERLENGTH + msg.pdl));  // Excluding the checksum
	buf[3]  = msg.targetRdmId[0];
	buf[4]  = msg.targetRdmId[1];
	buf[5]  = msg.targetRdmId[2];
	buf[6]  = msg.targetRdmId[3];
	buf[7]  = msg.targetRdmId[4];
	buf[8]  = msg.targetRdmId[5];
	buf[9]  = msg.sourceRdmId[0];
	buf[10] = msg.sourceRdmId[1];
	buf[11] = msg.sourceRdmId[2];
	buf[12] = msg.sourceRdmId[3];
	buf[13] = msg.sourceRdmId[4];
	buf[14] = msg.sourceRdmId[5];
	buf[15] = msg.tn;
	if(msg.portId != 0)
	{
		buf[16] = (msg.portId);  // Port ID
	}
	else
	{
		buf[16] = (1);  // Port ID is valid from 1 to 255
	}
	if(msg.cc == RDM_GET_COMMAND_RESPONSE || msg.cc == RDM_SET_COMMAND_RESPONSE)
	{
		buf[17] = (msg.msgCount);  // Message Count
	}
	else
	{
		buf[17] = (0);  // Controller should not send a message count
	}
	buf[18] = (0);  // Subdevice Field
	buf[19] = (0);  // Subdevice Field
	buf[20] = (msg.cc);
	buf[21] = (uint8_t)((msg.pid_num >> 8) & 0xFF);
	buf[22] = (uint8_t)((msg.pid_num >> 0) & 0xFF);
	buf[23] = (msg.pdl);

	// Set Response type if response message
	if(msg.cc == RDM_DISCOVERY_COMMAND_RESPONSE || msg.cc == RDM_GET_COMMAND_RESPONSE ||
	   msg.cc == RDM_SET_COMMAND_RESPONSE)
	{
		buf[16] = msg.rt;
	}

	memcpy(&buf[24], data, msg.pdl);
	uint16_t checksum = 0;
	for(uint16_t i = 0; i < 24 + msg.pdl; i++)
	{
		checksum += buf[i];
	}
	buf[24 + msg.pdl] = (uint8_t)((checksum >> 8) & 0xFF);
	buf[25 + msg.pdl] = (uint8_t)((checksum >> 0) & 0xFF);
}

bool DecodeRdmMessage(uint8_t *msg, uint16_t length, rdm_message_header_t *buf)
{
	// Check minimum length and startcode
	if(length <= RDMHEADERLENGTH || msg[0] != SCRDM || msg[1] != SCSUBMESSAGE)
	{
          #ifdef RTTLOGGINGSET
                  LOGGER_ERROR("Lenght %d, SC: %d, SCSubM %d", length,msg[0],msg[1] );
          #endif
          return false;
	}
	uint16_t messageLength = msg[2];
	if(messageLength + 2 > length)
	{
		// Too less input data. Abort decoding
                #ifdef RTTLOGGINGSET
                  LOGGER_ERROR("Less Input Data");
                #endif
		return false;
	}
	// Check checksum
	uint16_t checksum = 0;
	for(uint16_t i = 0; i < messageLength; i++)
	{
		checksum += msg[i];
	}
	if(checksum != (msg[messageLength + 1] + (msg[messageLength] << 8)))
	{
                #ifdef RTTLOGGINGSET
                  LOGGER_ERROR("Wrong Checksum");
                #endif
		return false;
	}

	buf->response = (msg[20] & 1) != 0;
	if((msg[20] & 0xFE) == RDM_GET_COMMAND)
	{
		buf->cc = RDM_GET_COMMAND;
	}
	else if((msg[20] & 0xFE) == RDM_SET_COMMAND)
	{
		buf->cc = RDM_SET_COMMAND;
	}
	else if((msg[20] & 0xFE) == RDM_DISCOVERY_COMMAND)
	{
		buf->cc = RDM_DISCOVERY_COMMAND;
	}
	else
	{
                #ifdef RTTLOGGINGSET
                LOGGER_ERROR("Wrong CC");
                #endif
		return false;
	}
	buf->tn = msg[15];
	if(buf->response)
	{
		if(msg[16] < RDM_UNKNOWN)
		{
			buf->rt = (rdm_response_t)msg[16];
		}
		else
		{
			buf->rt = RDM_UNKNOWN;
		}
	}
	else
	{
		buf->portId = msg[16];
		if(buf->portId == 0)
		{
                  #ifdef RTTLOGGINGSET
                  LOGGER_ERROR("Port Id not 0");
                  #endif
                  return false;
		}
		// Message count must be zero from a controller generated message
		if(msg[17] != 0)
		{
                  #ifdef RTTLOGGINGSET
                  LOGGER_ERROR("Message count not zero");
                  #endif
			return false;
		}
		buf->msgCount = 0;
	}
	DecodeRdmId(msg, length, buf);

	buf->pid_num = (uint16_t)((msg[21] << 8) + msg[22]);

	buf->pdl = msg[23];
	return true;
}

void DecodeRdmId(uint8_t *msg, uint16_t length, rdm_message_header_t *buf)
{
	if(length <= RDMHEADERLENGTH)
	{
		return;
	}
	buf->response                = (msg[20] & 1) != 0;
	buf->targetRdmId[0]          = msg[3];
	buf->targetRdmId[1]          = msg[4];
	buf->targetRdmId[2]          = msg[5];
	buf->targetRdmId[3]          = msg[6];
	buf->targetRdmId[4]          = msg[7];
	buf->targetRdmId[5]          = msg[8];
	buf->sourceRdmId[0]          = msg[9];
	buf->sourceRdmId[1]          = msg[10];
	buf->sourceRdmId[2]          = msg[11];
	buf->sourceRdmId[3]          = msg[12];
	buf->sourceRdmId[4]          = msg[13];
	buf->sourceRdmId[5]          = msg[14];
	buf->isBroadcast             = true;
	buf->isManufacturerBroadcast = true;
	for(uint8_t i = 0; i < 6; i++)
	{
		if(buf->targetRdmId[i] != 0xFF)
		{
			buf->isBroadcast = false;
		}
		if(i >= 2)
		{
			if(buf->targetRdmId[i] != 0xFF)
			{
				buf->isManufacturerBroadcast = false;
			}
		}
	}
}

rdm_command_class_t DecodeCommandClass(uint8_t *msg, uint16_t length)
{
	if(length <= RDMHEADERLENGTH || msg[0] != SCRDM || msg[1] != SCSUBMESSAGE)
	{
		return (rdm_command_class_t)0;
	}

	if((msg[20] & 0xFE) == RDM_GET_COMMAND)
	{
		return RDM_GET_COMMAND;
	}
	else if((msg[20] & 0xFE) == RDM_SET_COMMAND)
	{
		return RDM_SET_COMMAND;
	}
	else if((msg[20] & 0xFE) == RDM_DISCOVERY_COMMAND)
	{
		return RDM_DISCOVERY_COMMAND;
	}
	else
	{
		return 0;
	}
}

uint16_t DecodePid(uint8_t *msg, uint16_t length)
{
	if(length <= RDMHEADERLENGTH || msg[0] != SCRDM || msg[1] != SCSUBMESSAGE)
	{
		return (rdm_command_class_t)0;
	}

	return (uint16_t)((msg[21] << 8) + msg[22]);
}
