#include "libRdmInterface.h"

#include "string.h"
#ifdef RTTLOGGINGSET
#define LOGENABLE 0
#include "loggerservice.h"
#endif

static bool RdmMatch(rdm_library_instance_t *rdmInstance);
#ifdef RDM_SUPPORT_PROXY
static void RdmProxyProcess(rdm_library_instance_t *instance, const uint8_t *msg);
static bool RdmProxyMatch(rdm_library_instance_t *instance, uint8_t *index);
static void DecodeRdmProxyDiscovery(rdm_library_instance_t *rdmInstance, uint8_t *dataBuffer, uint8_t index);
static void DecodeGetProxiedDeviceCount(rdm_library_instance_t *rdmInstance, uint8_t *dataBuffer);
static void DecodeGetProxiedDevices(rdm_library_instance_t *rdmInstance, uint8_t *dataBuffer);
#endif
#ifdef RDM_SUPPORT_DISCOVERY
static void DecodeRdmDiscovery(rdm_library_instance_t *rdmInstance, uint8_t *dataBuffer);
#endif
static void DecodeRdmSet(rdm_library_instance_t *rdmInstance, uint8_t *dataBuffer);
static void DecodeRdmGet(rdm_library_instance_t *rdmInstance, uint8_t *dataBuffer);
static void SendNackResponse(rdm_library_instance_t *instance, rdm_nack_reasoncodes_t reason);
static void SendResponse(rdm_library_instance_t * instance, uint8_t *data, uint8_t pdl, rdm_response_t rt);
#ifdef RDM_SUPPORT_PROXY
static void SendAckTimerResponse(rdm_library_instance_t *rdmInstance, uint16_t timeDelay);
#endif
#ifdef RDM_SUPPORT_DISCOVERY
static void DecodeDiscoveryDiscUnMute(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeDiscoveryDiscMute(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeDiscoveryDiscUniqueBranch(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif

#ifdef RDM_SUPPORT_REQUIRED
static void DecodeGetDeviceInfo(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetIdentify(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetSoftwareVersionLabel(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif

#if defined(RDM_SUPPORT_CUSTOM_DFU) || defined(RDM_SUPPORT_BASIC)
static void DecodeGetDeviceModelDescription(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetManufacturerLabel(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif

#ifdef RDM_SUPPORT_BASIC
static void DecodeGetFactoryDefaults(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetProductDetail(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetDeviceLabel(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetBootSoftwareVersionLabel(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetDmxPersonality(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetDmxPersonalityDescription(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetDmxStartaddress(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetSensorDefinition(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetSensorValue(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetDeviceHours(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetLampHours(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetDevicePowerCycles(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetPowerState(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetSupportedParameter(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetParameterDescription(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetCustomPid(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetBootSoftwareID(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif

#ifdef RDM_SUPPORT_CUSTOM_IQ
static void DecodeGetIq(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetCustomError(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetCustomWarning(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif


#ifdef RDM_SUPPORT_LAMP
static void DecodeGetLampStrikes(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetLampState(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif

#ifdef RDM_SUPPORT_SELFTEST
static void DecodeGetPerformSelftest(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetSelfTestDescription(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif

#ifdef RDM_SUPPORT_LAMP
static void DecodeGetLampOnMode(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif

#ifdef RDM_SUPPORT_EXTENDED
static void DecodeGetLanguageCapabilities(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetLanguage(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetSlotInfo(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetSlotDescription(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetDefaultSlotValue(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetDisplayInvert(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetDisplayLevel(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetPanInvert(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetTiltInvert(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetRealTimeClock(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetPresetPlayback(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif

#ifdef RDM_SUPPORT_DIMMER
static void DecodeGetDimmerInfo(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetMinimumLevel(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetMaximumLevel(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetCurve(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetCurveDescription(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetOutputResponseTime(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetOutputResponseTimeDescription(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetModulationFequency(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetModulationFequencyDescription(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif

#ifdef RDM_SUPPORT_CUSTOM_LEDCAL
static void DecodeGetCustomAdjustLedCalibrationData(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetCustomAdjustLedCalibrationDebug(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif

#ifdef RDM_SUPPORT_CUSTOM_SERVICEHOURS
static void DecodeGetCustomServiceHours(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif

#ifdef RDM_SUPPORT_COLORCONVERTER
static void DecodeGetColorConverterLicense(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetColorConverterCalibration(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeGetColorConverterControl(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif

#ifdef RDM_SUPPORT_CUSTOM_DFU
static void DecodeGetCustomDfu(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif

#ifdef RDM_SUPPORT_REQUIRED
static void DecodeSetIdentify(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif

#ifdef RDM_SUPPORT_BASIC
static void DecodeSetFactoryDefaults(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeSetDeviceLabel(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeSetDmxPersonality(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeSetDmxStartaddress(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeSetResetDevice(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeSetPowerState(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeSetCustomPid(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif

#ifdef RDM_SUPPORT_CUSTOM_IQ
static void DecodeSetIq(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif

#ifdef RDM_SUPPORT_SELFTEST
static void DecodeSetPerformSelftest(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif

#ifdef RDM_SUPPORT_CUSTOM_LEDCAL
static void DecodeSetCustomAdjustLedCalibrationSend(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeSetCustomAdjustLedCalibrationData(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif

#ifdef RDM_SUPPORT_CUSTOM_SERVICEHOURS
static void DecodeSetCustomServiceHours(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif

#ifdef RDM_SUPPORT_EXTENDED
static void DecodeSetLanguage(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeSetDisplayInvert(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeSetDisplayLevel(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeSetPanInvert(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeSetTiltInvert(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeSetCapturePreset(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeSetRealTimeClock(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeSetPresetPlayback(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif

#ifdef RDM_SUPPORT_DIMMER
static void DecodeSetLampOnMode(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeSetMinimumLevel(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeSetMaximumLevel(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeSetCurve(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeSetOutputResponseTime(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeSetModulationFequency(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif

#ifdef RDM_SUPPORT_COLORCONVERTER
static void DecodeSetColorConverterLicense(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeSetColorConverterCalibration(rdm_library_instance_t *rdmInstance, uint8_t *msg);
static void DecodeSetColorConverterControl(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif

#ifdef RDM_SUPPORT_CUSTOM_DFU
static void DecodeSetCustomDfu(rdm_library_instance_t *rdmInstance, uint8_t *msg);
#endif

static uint16_t bswap16(uint16_t val);
static uint32_t bswap32(uint32_t val);
static uint64_t bswap64(uint64_t val);

bool rdm_init(rdm_library_instance_t * rdmInstance, const uint8_t *_rdm_id,
			  void (*rdm_answer_callback)(const uint8_t *msg, uint16_t length, bool sentBreak, uint8_t _id))
{
  if(rdmInstance == NULL)
  {
    return false;
  }
  memset((uint8_t*)rdmInstance, 0, sizeof(rdm_library_instance_t));
  rdmInstance->initialized   = true;
  rdmInstance->discoveryMute = false;
#ifdef RDM_SUPPORT_PROXY
  memset(rdmInstance->proxyDiscoveryMute, 0, MAX_PROXY_FIXTURES);
#endif
  memcpy(rdmInstance->rdm_id, _rdm_id, 6);
  rdmInstance->rdm_answer_handler = rdm_answer_callback;
  return true;
}

void rdm_setRdmId(rdm_library_instance_t * rdmInstance, const uint8_t *_rdm_id)
{
  if(rdmInstance == NULL)
  {
    return;
  }
  memcpy(rdmInstance->rdm_id, _rdm_id, 6);
}

#ifdef RDM_SUPPORT_PROXY
void rdm_setProxyRdmIds(rdm_library_instance_t * rdmInstance, const uint8_t *_rdm_ids, uint8_t length, uint16_t _averageResponseTime100ms,
						void (*rdm_proxy_transmit_callback)(const uint8_t *data, uint16_t length, uint8_t _id))
{
  if(rdmInstance == NULL)
  {
    return;
  }
    if(length > MAX_PROXY_FIXTURES)
    {
            length = MAX_PROXY_FIXTURES;
    }
    rdmInstance->proxyFixtureCount = 0;

    for(int i = 0; i < length; i++)
    {
            bool flagValid = false;
            bool flagSame  = true;
            for(int x = 0; x < 6; x++)
            {
                    if(_rdm_ids[i * 6 + x] > 0)
                    {
                            flagValid = true;
                    }
                    if(_rdm_ids[i * 6 + x] != rdmInstance->rdm_id[x])
                    {
                            flagSame = false;
                    }
            }
            if(flagValid && !flagSame)
            {
                    memcpy(rdmInstance->proxyrdmIds[rdmInstance->proxyFixtureCount], &_rdm_ids[i * 6], 6);
                    rdmInstance->proxyFixtureCount++;
            }
    }
    rdmInstance->rdm_proxy_transmit_handler = rdm_proxy_transmit_callback;
    rdmInstance->averageResponseTime100ms   = _averageResponseTime100ms;
    rdmInstance->proxyListUpdated = true;
#ifdef RTTLOGGINGSET
    LOGGER_INFO("Set RDM Proxy Count: %d", rdmInstance->proxyFixtureCount);
#endif
}

#endif

#if defined(RDM_SUPPORT_QUEUED_MESSAGES) || defined(RDM_SUPPORT_PROXY)
void rdm_registerQueueCallbacks(rdm_library_instance_t * rdmInstance, void (*rdm_queue_count_callback)(uint8_t *rdmId, uint8_t *responseQueueCount,
																 uint8_t _id),
								bool (*rdm_queue_callback)(uint8_t *rdmId, uint8_t *data, uint16_t *length,
														   uint8_t _id))
{
  if(rdmInstance == NULL)
  {
    return;
  }
	rdmInstance->rdm_queue_count_handler = rdm_queue_count_callback;
	rdmInstance->rdm_queue_handler       = rdm_queue_callback;
}
#endif

void rdm_decodeMessage(rdm_library_instance_t * rdmInstance, const uint8_t *msg, uint16_t length, uint8_t _id)
{
  if(rdmInstance == NULL)
  {
    return;
  }
	rdmInstance->actId = _id;
	if(!rdmInstance->initialized)
	{
#ifdef RTTLOGGINGSET
		LOGGER_ERROR("Rdm handler not initialized!");
#endif
		return;
	}

	if(!DecodeRdmMessage((uint8_t *)msg, length, (rdm_message_header_t *)&rdmInstance->actHeader))
	{
#ifdef RTTLOGGINGSET
		LOGGER_ERROR("Rdm message could not decoded! Length %d", length);
#endif
		return;
	}
	if(rdmInstance->actHeader.response)
	{
#ifdef RTTLOGGINGSET
		LOGGER_ERROR("Rdm message is response");
#endif
		return;
	}

	if(!RdmMatch(rdmInstance))
	{
#ifdef RDM_SUPPORT_PROXY
		RdmProxyProcess(rdmInstance, msg);
#else
#ifdef RTTLOGGINGSET
		LOGGER_INFO("Wrong Rdm Id");
#endif

#endif
		return;
	}

	switch(rdmInstance->actHeader.cc)
	{
#ifdef RDM_SUPPORT_DISCOVERY
	case RDM_DISCOVERY_COMMAND:
		DecodeRdmDiscovery(rdmInstance, (uint8_t *)&msg[RDMDATAPOSITION]);
		break;
#endif
	case RDM_GET_COMMAND:
		DecodeRdmGet(rdmInstance, (uint8_t *)&msg[RDMDATAPOSITION]);
		break;
	case RDM_SET_COMMAND:
		DecodeRdmSet(rdmInstance, (uint8_t *)&msg[RDMDATAPOSITION]);
		break;
	case RDM_DISCOVERY_COMMAND_RESPONSE:
	case RDM_SET_COMMAND_RESPONSE:
	case RDM_GET_COMMAND_RESPONSE:
	default:
		SendNackResponse(rdmInstance, NR_UNSUPPORTED_COMMAND_CLASS);
		break;
	}
}

#ifdef RDM_SUPPORT_DISCOVERY
static void DecodeRdmDiscovery(rdm_library_instance_t *rdmInstance, uint8_t *dataBuffer)
{
	if(rdmInstance->actHeader.cc != RDM_DISCOVERY_COMMAND)
	{
#ifdef RTTLOGGINGSET
		LOGGER_ERROR("Rdm decode error");
#endif
	}
	switch(rdmInstance->actHeader.pid_num)
	{
	case RDMPID_DISC_UNIQUE_BRANCH:
		DecodeDiscoveryDiscUniqueBranch(rdmInstance, dataBuffer);
		break;
	case RDMPID_DISC_MUTE:
		rdmInstance->discoveryMute = true;
		DecodeDiscoveryDiscMute(rdmInstance, dataBuffer);
		break;
	case RDMPID_DISC_UN_MUTE:
		rdmInstance->discoveryMute = false;
#ifdef RDM_SUPPORT_PROXY
		if(rdmInstance->actHeader.isBroadcast)
		{
			memset(rdmInstance->proxyDiscoveryMute, 0, MAX_PROXY_FIXTURES);
		}
#endif
		if(!(rdmInstance->actHeader.isBroadcast))
		{
			DecodeDiscoveryDiscUnMute(rdmInstance, dataBuffer);
		}
		break;
	default:
		SendNackResponse(rdmInstance, NR_UNKNOWN_PID);
		break;
	}
}
#endif

#ifdef RDM_SUPPORT_PROXY
static void DecodeRdmProxyDiscovery(rdm_library_instance_t *rdmInstance, uint8_t *dataBuffer, uint8_t index)
{
	if(rdmInstance->actHeader.cc != RDM_DISCOVERY_COMMAND)
	{
#ifdef RTTLOGGINGSET
		LOGGER_ERROR("Rdm decode error");
#endif
	}
	switch(rdmInstance->actHeader.pid_num)
	{
	case RDMPID_DISC_UNIQUE_BRANCH:
		// Should not occur
		break;
	case RDMPID_DISC_MUTE:
		rdmInstance->proxyDiscoveryMute[index] = true;
		DecodeDiscoveryDiscMute(rdmInstance, dataBuffer);
		break;
	case RDMPID_DISC_UN_MUTE:
		rdmInstance->proxyDiscoveryMute[index] = false;
		DecodeDiscoveryDiscUnMute(rdmInstance, dataBuffer);
		break;
	default:
		SendNackResponse(rdmInstance, NR_UNKNOWN_PID);
		break;
	}
}
#endif

static void DecodeRdmGet(rdm_library_instance_t *rdmInstance, uint8_t *dataBuffer)
{
	if(rdmInstance->actHeader.cc != RDM_GET_COMMAND)
	{
#ifdef RTTLOGGINGSET
		LOGGER_ERROR("Rdm decode error");
#endif
	}
	switch(rdmInstance->actHeader.pid_num)
	{
#ifdef RDM_SUPPORT_REQUIRED
	case RDMPID_DEVICEINFO:
		DecodeGetDeviceInfo(rdmInstance, dataBuffer);
		break;
	case RDMPID_SOFTWARE_VERSION_LABEL:
		DecodeGetSoftwareVersionLabel(rdmInstance, dataBuffer);
		break;
	case RDMPID_IDENTIFY_DEVICE:
		DecodeGetIdentify(rdmInstance, dataBuffer);
		break;
#endif
#if defined(RDM_SUPPORT_CUSTOM_DFU) || defined(RDM_SUPPORT_BASIC)
	case RDMPID_DEVICEMODELDESCRIPTION:
			DecodeGetDeviceModelDescription(rdmInstance, dataBuffer);
			break;
		case RDMPID_MANUFACTURER_LABEL:
			DecodeGetManufacturerLabel(rdmInstance, dataBuffer);
			break;
#endif
#ifdef RDM_SUPPORT_BASIC
	case RDMPID_PRODUCTDETAILIDLIST:
		DecodeGetProductDetail(rdmInstance, dataBuffer);
		break;

	case RDMPID_DEVICE_LABEL:
		DecodeGetDeviceLabel(rdmInstance, dataBuffer);
		break;
	case RDMPID_FACTORY_DEFAULTS:
		DecodeGetFactoryDefaults(rdmInstance, dataBuffer);
		break;
	case RDMPID_BOOT_SOFTWARE_VERSION_LABEL:
		DecodeGetBootSoftwareVersionLabel(rdmInstance, dataBuffer);
		break;
	case RDMPID_DMX_PERSONALITY:
		DecodeGetDmxPersonality(rdmInstance, dataBuffer);
		break;
	case RDMPID_DMX_PERSONALITY_DESCRIPTION:
		DecodeGetDmxPersonalityDescription(rdmInstance, dataBuffer);
		break;
	case RDMPID_DMXSTARTADDRESS:
		DecodeGetDmxStartaddress(rdmInstance, dataBuffer);
		break;
	case RDMPID_SENSOR_DEFINITION:
		DecodeGetSensorDefinition(rdmInstance, dataBuffer);
		break;
	case RDMPID_SENSOR_VALUE:
		DecodeGetSensorValue(rdmInstance, dataBuffer);
		break;
	case RDMPID_DEVICE_HOURS:
		DecodeGetDeviceHours(rdmInstance, dataBuffer);
		break;
	case RDMPID_LAMP_HOURS:
		DecodeGetLampHours(rdmInstance, dataBuffer);
		break;
	case RDMPID_DEVICE_POWER_CYCLES:
		DecodeGetDevicePowerCycles(rdmInstance, dataBuffer);
		break;
	case RDMPID_POWER_STATE:
		DecodeGetPowerState(rdmInstance, dataBuffer);
		break;
	case RDMPID_SUPPORTED_PARAMETER:
		DecodeGetSupportedParameter(rdmInstance, dataBuffer);
		break;
	case RDMPID_PARAMETER_DESCRIPTION:
		DecodeGetParameterDescription(rdmInstance, dataBuffer);
		break;
	case RDMPID_BOOT_SOFTWARE_VERSION_ID:
		DecodeGetBootSoftwareID(rdmInstance, dataBuffer);
		break;
#endif
#ifdef RDM_SUPPORT_LAMP
	case RDMPID_LAMP_STRIKES:
		DecodeGetLampStrikes(rdmInstance, dataBuffer);
		break;
	case RDMPID_LAMP_STATE:
		DecodeGetLampState(rdmInstance, dataBuffer);
		break;
#endif
#ifdef RDM_SUPPORT_SELFTEST
	case RDMPID_PERFORM_SELFTEST:
		DecodeGetPerformSelftest(rdmInstance, dataBuffer);
		break;
	case RDMPID_SELF_TEST_DESCRIPTION:
		DecodeGetSelfTestDescription(rdmInstance, dataBuffer);
		break;
#endif
#ifdef RDM_SUPPORT_CUSTOM_IQ
	case RDMCPID_IQ:
		DecodeGetIq(rdmInstance, dataBuffer);
		break;
	case RDMCPID_CUSTOM_ERROR:
		DecodeGetCustomError(rdmInstance, dataBuffer);
		break;
	case RDMCPID_CUSTOM_WARNING:
		DecodeGetCustomWarning(rdmInstance, dataBuffer);
		break;
#endif
#ifdef RDM_SUPPORT_CUSTOM_LEDCAL
	case RDMCPID_CUSTOM_ADJUSTLEDCALIBRATIONDATA:
		DecodeGetCustomAdjustLedCalibrationData(rdmInstance, dataBuffer);
		break;
	case RDMCPID_CUSTOM_ADJUSTLEDCALIBRATIONDEBUG:
		DecodeGetCustomAdjustLedCalibrationDebug(rdmInstance, dataBuffer);
		break;
#endif
#ifdef RDM_SUPPORT_CUSTOM_SERVICEHOURS
	case RDMCPID_CUSTOM_SERVICEHOURS:
		DecodeGetCustomServiceHours(rdmInstance, dataBuffer);
		break;
#endif
#ifdef RDM_SUPPORT_EXTENDED
	case RDMPID_LANGUAGE_CAPABILITIES:
		DecodeGetLanguageCapabilities(rdmInstance, dataBuffer);
		break;
	case RDMPID_LANGUAGE:
		DecodeGetLanguage(rdmInstance, dataBuffer);
		break;

	case RDMPID_SLOT_INFO:
		DecodeGetSlotInfo(rdmInstance, dataBuffer);
		break;
	case RDMPID_SLOT_DESCRIPTION:
		DecodeGetSlotDescription(rdmInstance, dataBuffer);
		break;
	case RDMPID_DEFAULT_SLOT_VALUE:
		DecodeGetDefaultSlotValue(rdmInstance, dataBuffer);
		break;
	case RDMPID_DISPLAY_INVERT:
		DecodeGetDisplayInvert(rdmInstance, dataBuffer);
		break;
	case RDMPID_DISPLAY_LEVEL:
		DecodeGetDisplayLevel(rdmInstance, dataBuffer);
		break;
	case RDMPID_PAN_INVERT:
		DecodeGetPanInvert(rdmInstance, dataBuffer);
		break;
	case RDMPID_TILT_INVERT:
		DecodeGetTiltInvert(rdmInstance, dataBuffer);
		break;
	case RDMPID_REAL_TIME_CLOCK:
		DecodeGetRealTimeClock(rdmInstance, dataBuffer);
		break;
	case RDMPID_PRESET_PLAYBACK:
		DecodeGetPresetPlayback(rdmInstance, dataBuffer);
		break;
#endif
#ifdef RDM_SUPPORT_LAMP
	case RDMPID_LAMP_ON_MODE:
		DecodeGetLampOnMode(rdmInstance, dataBuffer);
		break;
#endif
#ifdef RDM_SUPPORT_DIMMER
	case RDMPID_DIMMER_INFO:
		DecodeGetDimmerInfo(rdmInstance, dataBuffer);
		break;
	case RDMPID_MINIMUM_LEVEL:
		DecodeGetMinimumLevel(rdmInstance, dataBuffer);
		break;
	case RDMPID_MAXIMUM_LEVEL:
		DecodeGetMaximumLevel(rdmInstance, dataBuffer);
		break;
	case RDMPID_CURVE:
		DecodeGetCurve(rdmInstance, dataBuffer);
		break;
	case RDMPID_CURVE_DESCRIPTION:
		DecodeGetCurveDescription(rdmInstance, dataBuffer);
		break;
	case RDMPID_OUTPUT_RESPONSE_TIME:
		DecodeGetOutputResponseTime(rdmInstance, dataBuffer);
		break;
	case RDMPID_OUTPUT_RESPONSE_TIME_DESCRIPTION:
		DecodeGetOutputResponseTimeDescription(rdmInstance, dataBuffer);
		break;
	case RDMPID_MODULATION_FREQUENCY:
		DecodeGetModulationFequency(rdmInstance, dataBuffer);
		break;
	case RDMPID_MODULATION_FREQUENCY_DESCRIPTION:
		DecodeGetModulationFequencyDescription(rdmInstance, dataBuffer);
		break;
#endif
#ifdef RDM_SUPPORT_COLORCONVERTER
	case RDMCPID_COLORCONVERTER_LICENSE:
		DecodeGetColorConverterLicense(rdmInstance, dataBuffer);
		break;
	case RDMCPID_COLORCONVERTER_CALIBRATION:
		DecodeGetColorConverterCalibration(rdmInstance, dataBuffer);
		break;
	case RDMCPID_COLORCONVERTER_CONTROL:
		DecodeGetColorConverterControl(rdmInstance, dataBuffer);
		break;
#endif
#ifdef RDM_SUPPORT_QUEUED_MESSAGES
	case RDMPID_QUEUED_MESSAGE:
		DecodeGetQueuedMessage(rdmInstance, dataBuffer);
		break;
#endif
#ifdef RDM_SUPPORT_PROXY
	case RDMPID_PROXIED_DEVICE_COUNT:
		DecodeGetProxiedDeviceCount(rdmInstance, dataBuffer);
		break;
        case RDMPID_PROXIED_DEVICES:
		DecodeGetProxiedDevices(rdmInstance, dataBuffer);
		break;
#endif
#ifdef RDM_SUPPORT_CUSTOM_DFU
	case RDMCPID_DFU:
		DecodeGetCustomDfu(rdmInstance, dataBuffer);
		break;
#endif
	default:
#ifdef RDM_SUPPORT_BASIC
		DecodeGetCustomPid(rdmInstance, dataBuffer);
		break;
#else
		SendNackResponse(rdmInstance, NR_UNSUPPORTED_COMMAND_CLASS);
break;
#endif
	}
}

static void DecodeRdmSet(rdm_library_instance_t *rdmInstance, uint8_t *dataBuffer)
{
	switch(rdmInstance->actHeader.pid_num)
	{
#ifdef RDM_SUPPORT_REQUIRED
	case RDMPID_IDENTIFY_DEVICE:
                DecodeSetIdentify(rdmInstance, dataBuffer);
                break;
#endif
#ifdef RDM_SUPPORT_BASIC
	case RDMPID_DEVICE_LABEL:
		DecodeSetDeviceLabel(rdmInstance, dataBuffer);
		break;
	case RDMPID_FACTORY_DEFAULTS:
		DecodeSetFactoryDefaults(rdmInstance, dataBuffer);
		break;
	case RDMPID_DMX_PERSONALITY:
		DecodeSetDmxPersonality(rdmInstance, dataBuffer);
		break;
	case RDMPID_DMXSTARTADDRESS:
		DecodeSetDmxStartaddress(rdmInstance, dataBuffer);
		break;
	case RDMPID_RESET_DEVICE:
		DecodeSetResetDevice(rdmInstance, dataBuffer);
		break;
	case RDMPID_POWER_STATE:
		DecodeSetPowerState(rdmInstance, dataBuffer);
		break;
#endif
#ifdef RDM_SUPPORT_SELFTEST
	case RDMPID_PERFORM_SELFTEST:
		DecodeSetPerformSelftest(rdmInstance, dataBuffer);
		break;
#endif
#ifdef RDM_SUPPORT_CUSTOM_IQ
	case RDMCPID_IQ:
		DecodeSetIq(rdmInstance, dataBuffer);
		break;
#endif
#ifdef RDM_SUPPORT_CUSTOM_LEDCAL
	case RDMCPID_CUSTOM_ADJUSTLEDCALIBRATIONSEND:
		DecodeSetCustomAdjustLedCalibrationSend(rdmInstance, dataBuffer);
		break;
	case RDMCPID_CUSTOM_ADJUSTLEDCALIBRATIONDATA:
		DecodeSetCustomAdjustLedCalibrationData(rdmInstance, dataBuffer);
		break;
#endif
#ifdef RDM_SUPPORT_CUSTOM_SERVICEHOURS
	case RDMCPID_CUSTOM_SERVICEHOURS:
		DecodeSetCustomServiceHours(rdmInstance, dataBuffer);
		break;
#endif
#ifdef RDM_SUPPORT_LAMP
	case RDMPID_LAMP_ON_MODE:
		DecodeSetLampOnMode(rdmInstance, dataBuffer);
		break;
#endif
#ifdef RDM_SUPPORT_EXTENDED
	case RDMPID_LANGUAGE:
		DecodeSetLanguage(rdmInstance, dataBuffer);
		break;
	case RDMPID_DISPLAY_INVERT:
		DecodeSetDisplayInvert(rdmInstance, dataBuffer);
		break;
	case RDMPID_DISPLAY_LEVEL:
		DecodeSetDisplayLevel(rdmInstance, dataBuffer);
		break;
	case RDMPID_PAN_INVERT:
		DecodeSetPanInvert(rdmInstance, dataBuffer);
		break;
	case RDMPID_TILT_INVERT:
		DecodeSetTiltInvert(rdmInstance, dataBuffer);
		break;
	case RDMPID_CAPTURE_PRESET:
		DecodeSetCapturePreset(rdmInstance, dataBuffer);
		break;
	case RDMPID_REAL_TIME_CLOCK:
		DecodeSetRealTimeClock(rdmInstance, dataBuffer);
		break;
	case RDMPID_PRESET_PLAYBACK:
		DecodeSetPresetPlayback(rdmInstance, dataBuffer);
		break;
#endif
#ifdef RDM_SUPPORT_DIMMER
	case RDMPID_MINIMUM_LEVEL:
		DecodeSetMinimumLevel(rdmInstance, dataBuffer);
		break;
	case RDMPID_MAXIMUM_LEVEL:
		DecodeSetMaximumLevel(rdmInstance, dataBuffer);
		break;
	case RDMPID_CURVE:
		DecodeSetCurve(rdmInstance, dataBuffer);
		break;
	case RDMPID_OUTPUT_RESPONSE_TIME:
		DecodeSetOutputResponseTime(rdmInstance, dataBuffer);
		break;
	case RDMPID_MODULATION_FREQUENCY:
		DecodeSetModulationFequency(rdmInstance, dataBuffer);
		break;
#endif
#ifdef RDM_SUPPORT_COLORCONVERTER
	case RDMCPID_COLORCONVERTER_LICENSE:
		DecodeSetColorConverterLicense(rdmInstance, dataBuffer);
		break;
	case RDMCPID_COLORCONVERTER_CALIBRATION:
		DecodeSetColorConverterCalibration(rdmInstance, dataBuffer);
		break;
	case RDMCPID_COLORCONVERTER_CONTROL:
		DecodeSetColorConverterControl(rdmInstance, dataBuffer);
		break;
#endif
#ifdef RDM_SUPPORT_CUSTOM_DFU
	case RDMCPID_DFU:
		DecodeSetCustomDfu(rdmInstance, dataBuffer);
		break;
#endif
	case RDMPID_DEVICEINFO:
	case RDMPID_SUPPORTED_PARAMETER:
		SendNackResponse(rdmInstance, NR_UNSUPPORTED_COMMAND_CLASS);
		break;
	default:
#ifdef RDM_SUPPORT_BASIC
		DecodeSetCustomPid(rdmInstance, dataBuffer);
		break;
#else
		SendNackResponse(rdmInstance, NR_UNSUPPORTED_COMMAND_CLASS);
	break;
#endif
	}
}

#ifdef RDM_SUPPORT_DISCOVERY

static void DecodeDiscoveryDiscUniqueBranch(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	if(rdmInstance->actHeader.pdl >= sizeof(rdm_disc_unique_branch_receive_t))
	{
		rdm_disc_unique_branch_receive_t *branchData = (rdm_disc_unique_branch_receive_t *)msg;
		uint8_t actRdmId[6];
		uint64_t rmdId_number = (uint64_t)rdmInstance->rdm_id[5] | ((uint64_t)rdmInstance->rdm_id[4] << 8) | ((uint64_t)rdmInstance->rdm_id[3] << 16) |
								((uint64_t)rdmInstance->rdm_id[2] << 24) | ((uint64_t)rdmInstance->rdm_id[1] << 32) | ((uint64_t)rdmInstance->rdm_id[0] << 40);
		uint64_t upperBound_number =
			(uint64_t)branchData->upperBoundUid[5] | ((uint64_t)branchData->upperBoundUid[4] << 8) |
			((uint64_t)branchData->upperBoundUid[3] << 16) | ((uint64_t)branchData->upperBoundUid[2] << 24) |
			((uint64_t)branchData->upperBoundUid[1] << 32) | ((uint64_t)branchData->upperBoundUid[0] << 40);
		uint64_t lowerBound_number =
			(uint64_t)branchData->lowerBoundUid[5] | ((uint64_t)branchData->lowerBoundUid[4] << 8) |
			((uint64_t)branchData->lowerBoundUid[3] << 16) | ((uint64_t)branchData->lowerBoundUid[2] << 24) |
			((uint64_t)branchData->lowerBoundUid[1] << 32) | ((uint64_t)branchData->lowerBoundUid[0] << 40);
		bool match = false;
		if(rmdId_number <= upperBound_number && rmdId_number >= lowerBound_number && !(rdmInstance->discoveryMute))
		{
			match = true;
			memcpy(actRdmId, rdmInstance->rdm_id, 6);
		}
#ifdef RDM_SUPPORT_PROXY
		else
		{
			for(uint8_t i = 0; i < rdmInstance->proxyFixtureCount; i++)
			{
				if(!(rdmInstance->proxyDiscoveryMute[i]))
				{
					rmdId_number = (uint64_t)rdmInstance->proxyrdmIds[i][5] | ((uint64_t)rdmInstance->proxyrdmIds[i][4] << 8) |
								   ((uint64_t)rdmInstance->proxyrdmIds[i][3] << 16) | ((uint64_t)rdmInstance->proxyrdmIds[i][2] << 24) |
								   ((uint64_t)rdmInstance->proxyrdmIds[i][1] << 32) | ((uint64_t)rdmInstance->proxyrdmIds[i][0] << 40);
					if(rmdId_number <= upperBound_number && rmdId_number >= lowerBound_number)
					{
						match = true;
						memcpy(actRdmId, rdmInstance->proxyrdmIds[i], 6);
						break;
					}
				}
			}
		}
#endif
		if(match)
		{
			rdm_disc_unique_branch_response_t response;
			memset(response.responsePreamble, 0xFE, 7);
			response.preambleSepearator = 0xAA;

			response.eUid[0] = actRdmId[0] | 0xAA;
			response.eUid[1] = actRdmId[0] | 0x55;

			response.eUid[2] = actRdmId[1] | 0xAA;
			response.eUid[3] = actRdmId[1] | 0x55;

			response.eUid[4] = actRdmId[2] | 0xAA;
			response.eUid[5] = actRdmId[2] | 0x55;

			response.eUid[6] = actRdmId[3] | 0xAA;
			response.eUid[7] = actRdmId[3] | 0x55;

			response.eUid[8] = actRdmId[4] | 0xAA;
			response.eUid[9] = actRdmId[4] | 0x55;

			response.eUid[10] = actRdmId[5] | 0xAA;
			response.eUid[11] = actRdmId[5] | 0x55;

			uint16_t checksum = 0;
			for(uint8_t i = 0; i < sizeof(response.eUid); i++)
			{
				checksum += response.eUid[i];
			}
			uint8_t msbChecksum   = (uint8_t)(checksum >> 8);
			uint8_t lsbChecksum   = (uint8_t)checksum;
			response.eChecksum[0] = msbChecksum | 0xAA;
			response.eChecksum[1] = msbChecksum | 0x55;
			response.eChecksum[2] = lsbChecksum | 0xAA;
			response.eChecksum[3] = lsbChecksum | 0x55;
                        if(rdmInstance->rdm_answer_handler != NULL)
                        {
                          rdmInstance->rdm_answer_handler((uint8_t *)&response, sizeof(response), false, rdmInstance->actId);
                        }
		}
	}
}

static void DecodeDiscoveryDiscMute(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_disc_unmute_response_t response;
        memset(&response, 0, sizeof(response));
        #ifdef RDM_SUPPORT_PROXY
          if(rdmInstance->actIsProxy)
          {
            response.controlField_managedProxyFlag = 1;
          }
          else
          {
            response.controlField_proxiedDeviceFlag = 1;
          }
        #endif

	SendResponse(rdmInstance, (uint8_t *)&response, sizeof(response), RDM_ACK);
	(void)msg;
}

static void DecodeDiscoveryDiscUnMute(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_disc_unmute_response_t response;
	memset(&response, 0, sizeof(response));
         #ifdef RDM_SUPPORT_PROXY
          if(rdmInstance->actIsProxy)
          {
            response.controlField_managedProxyFlag = 1;
          }
          else
          {
            response.controlField_proxiedDeviceFlag = 1;
          }
        #endif
	SendResponse(rdmInstance, (uint8_t *)&response, sizeof(response), RDM_ACK);

	(void)msg;
}

#endif

#ifdef RDM_SUPPORT_REQUIRED
static void DecodeGetDeviceInfo(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_device_info_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetDeviceInfo(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		response.protocollVersion  = bswap16(response.protocollVersion);
		response.deviceModelId     = bswap16(response.deviceModelId);
		response.productCategory   = bswap16(response.productCategory);
		response.softwareVersionId = bswap32(response.softwareVersionId);
		response.dmxFootprint      = bswap16(response.dmxFootprint);
		response.dmxStartaddress   = bswap16(response.dmxStartaddress);
		response.SubDeviceCount    = bswap16(response.SubDeviceCount);
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(response), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetSoftwareVersionLabel(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_softwareversion_label_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetSoftwareVersionLabel(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&response, response.labelSize, implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetIdentify(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_identify_device_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetIdentify(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(response), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}
#endif

#if defined(RDM_SUPPORT_CUSTOM_DFU) || defined(RDM_SUPPORT_BASIC)
static void DecodeGetDeviceModelDescription(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_device_model_description_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetDeviceModelDescription(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&response, response.labelSize, implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetManufacturerLabel(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_manufacturer_label_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetManufacturerLabel(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&response, response.labelSize, implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

#endif

#ifdef RDM_SUPPORT_BASIC
static void DecodeGetProductDetail(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_product_detail_id_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetProductDetail(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		for(uint8_t i = 0; i < response.productDetailsCount; i++)
		{
			response.productDetails[i] = bswap16(response.productDetails[i]);
		}
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(uint16_t) * response.productDetailsCount,
					 implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetFactoryDefaults(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_factory_default_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetFactoryDefaults(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(response), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetDeviceLabel(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_device_label_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetDeviceLabel(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		if(response.labelSize > RDMLABELLENGTH)
		{
			response.labelSize = RDMLABELLENGTH;
		}
		SendResponse(rdmInstance, (uint8_t *)&response, response.labelSize, implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetBootSoftwareVersionLabel(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_boot_softwareversion_label_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetBootSoftwareVersionLabel(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&response, response.labelSize, implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetDmxPersonality(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_dmx_personality_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetDmxPersonality(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(response), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetDmxPersonalityDescription(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_dmx_personality_description_get_response_t response;
	memset(&response, 0, sizeof(response));
	if(rdmInstance->actHeader.pdl >= 1)
	{
		response.requestedPersonality = msg[0];
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
		return;
	}
	RdmExtGetDmxPersonalityDescription(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		response.footprint = bswap16(response.footprint);
		SendResponse(rdmInstance, (uint8_t *)&response, 3 + response.labelSize, implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetDmxStartaddress(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_dmx_startaddress_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetDmxStartaddress(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		response.address = bswap16(response.address);
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(response), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetSensorDefinition(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_sensor_defintion_get_response_t response;
	memset(&response, 0, sizeof(response));
	if(rdmInstance->actHeader.pdl >= 1)
	{
		response.requestedSensor = msg[0];
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
		return;
	}
	RdmExtGetSensorDefinition(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		response.rangeMinimumValue  = (int16_t)bswap16((uint16_t)response.rangeMinimumValue);
		response.rangeMaximumValue  = (int16_t)bswap16((uint16_t)response.rangeMaximumValue);
		response.normalMinimumValue = (int16_t)bswap16((uint16_t)response.normalMinimumValue);
		response.normalMaximumValue = (int16_t)bswap16((uint16_t)response.normalMaximumValue);
		SendResponse(rdmInstance, (uint8_t *)&response, 13 + response.labelSize, implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
}

static void DecodeGetSensorValue(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_sensor_value_get_response_t response;
	memset(&response, 0, sizeof(response));
	if(rdmInstance->actHeader.pdl >= 1)
	{
		response.requestedSensor = msg[0];
	}
	RdmExtGetSensorValue(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		response.presentValue         = bswap16(response.presentValue);
		response.lowestDetectedValue  = bswap16(response.lowestDetectedValue);
		response.highestDetectedValue = bswap16(response.highestDetectedValue);
		response.recordedValue        = bswap16(response.recordedValue);
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(response), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
}

static void DecodeGetDeviceHours(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_device_hours_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetDeviceHours(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		response.deviceHours = bswap32(response.deviceHours);
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(response), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetLampHours(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_lamphours_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetLampHours(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		response.lampHours = bswap32(response.lampHours);
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(response), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetDevicePowerCycles(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_device_power_cycle_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetDevicePowerCycles(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		response.powerCycleCount = bswap32(response.powerCycleCount);
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(response), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetPowerState(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_power_state_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetPowerState(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(response), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetSupportedParameter(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	uint16_t responseBE[RDMDATALENGTHMAX / 2];
	uint8_t pidCount = 0;
	memset(&responseBE, 0, sizeof(responseBE));
	RdmExtGetSupportedParameter(&implemented, responseBE, &pidCount);
	if(implemented.response != RDM_NACK_REASON)
	{
		if(pidCount >= RDMDATALENGTHMAX / 2)
		{
			pidCount = RDMDATALENGTHMAX / 2;
		}
		for(uint8_t i = 0; i < pidCount; i++)
		{
			responseBE[i] = bswap16(responseBE[i]);
		}
		SendResponse(rdmInstance, (uint8_t *)&responseBE, pidCount * 2, implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetParameterDescription(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_parameter_description_get_response_t response;
	memset(&response, 0, sizeof(response));
	// Set Id
	if(rdmInstance->actHeader.pdl >= 2)
	{
		uint16_t tmp;
		memcpy(&tmp, msg, sizeof(tmp));
		response.requestedParameter = bswap16(tmp);
		RdmExtGetParameterDescription(&implemented, &response);
		if(implemented.response != RDM_NACK_REASON)
		{
			response.requestedParameter = bswap16(response.requestedParameter);
			response.defaultValue       = bswap32(response.defaultValue);
			response.maxValidValue      = bswap32(response.maxValidValue);
			response.minValidValue      = bswap32(response.minValidValue);
                        if(response.labelSize > RDMLABELLENGTH)
                        {
                          response.labelSize = RDMLABELLENGTH;
                        }
			SendResponse(rdmInstance, (uint8_t *)&response, sizeof(response) - RDMLABELLENGTH - sizeof(response.labelSize) + response.labelSize, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}

static void DecodeGetBootSoftwareID(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response   = RDM_ACK;
	implemented.reason     = NR_UNKNOWN_PID;
	uint32_t boot_software = 0;
	RdmExtGetBootSoftwareID(&implemented, &boot_software);
	if(implemented.response != RDM_NACK_REASON)
	{
		boot_software = bswap32(boot_software);
		SendResponse(rdmInstance, (uint8_t *)&boot_software, sizeof(uint32_t), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetCustomPid(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	uint8_t response[RDMDATALENGTHMAX];
	memset(&response, 0, sizeof(response));
	uint16_t responseSize = 0;
	RdmExtGetCustomPid(rdmInstance->actHeader.pid_num, msg, rdmInstance->actHeader.pdl, &implemented, response, &responseSize);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&response, (uint8_t)responseSize, implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
}
#endif

#ifdef RDM_SUPPORT_LAMP
static void DecodeGetLampStrikes(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_lampstrike_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetLampStrikes(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		response.lampStrikes = bswap32(response.lampStrikes);
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(response), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetLampState(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_lampstate_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetLampState(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(response), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetLampOnMode(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_lamp_on_mode_response_t response;
	RdmExtGetLampOnMode(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(rdm_lamp_on_mode_response_t), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}
#endif

#ifdef RDM_SUPPORT_SELFTEST
static void DecodeGetPerformSelftest(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_perform_selftest_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetPerformSelftest(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(response), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetSelfTestDescription(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_selftest_description_get_response_t response;
	memset(&response, 0, sizeof(response));
	if(rdmInstance->actHeader.pdl >= 1)
	{
		response.requestedSelftest = msg[0];
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
		return;
	}
	RdmExtGetSelfTestDescription(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&response, 1 + response.labelSize, implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}
#endif

#ifdef RDM_SUPPORT_CUSTOM_IQ
static void DecodeGetIq(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_iq_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetIq(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		response.deviceModelId        = bswap16(response.deviceModelId);
		response.serial               = bswap64(response.serial);
		response.dmxStartaddress      = bswap16(response.dmxStartaddress);
		response.dmxFootprint         = bswap16(response.dmxFootprint);
		response.supportedInputSource = bswap16(response.supportedInputSource);
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(response), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetCustomError(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_custom_label_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetCustomError(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&response, response.labelSize, implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetCustomWarning(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_custom_label_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetCustomWarning(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&response, response.labelSize, implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}
#endif

#ifdef RDM_SUPPORT_EXTENDED
static void DecodeGetLanguageCapabilities(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	uint8_t response[RDMDATALENGTHMAX];
	uint8_t languageCount = 0;
	memset(&response, 0, sizeof(response));
	RdmExtGetLanguageCapabilities(&implemented, response, &languageCount);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&response, languageCount * 2, implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetLanguage(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_language_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetLanguage(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(rdm_language_get_response_t), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetSlotInfo(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	uint8_t responseSlot[RDMDATALENGTHMAX];
	uint8_t slotCount = 0;
	memset(&responseSlot, 0, sizeof(responseSlot));
	RdmExtGetSlotInfo(&implemented, (rdm_slot_info_get_response_array_t *)responseSlot, &slotCount);
	if(implemented.response != RDM_NACK_REASON)
	{
		for(uint8_t i = 0; i < slotCount; i++)
		{
			uint16_t slotOffset  = ((rdm_slot_info_get_response_array_t *)responseSlot)[i].slotOffset;
			uint16_t slotLabelId = ((rdm_slot_info_get_response_array_t *)responseSlot)[i].slotLabelId;
			slotOffset           = bswap16(slotOffset);
			slotLabelId          = bswap16(slotLabelId);
			memcpy((uint16_t *)&(((rdm_slot_info_get_response_array_t *)responseSlot)[i].slotOffset), &slotOffset, 2);
			memcpy((uint16_t *)&(((rdm_slot_info_get_response_array_t *)responseSlot)[i].slotLabelId), &slotLabelId, 2);
		}
		SendResponse(rdmInstance, (uint8_t *)&responseSlot, slotCount * sizeof(rdm_slot_info_get_response_array_t),
					 implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetSlotDescription(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_slot_desc_get_response_t response;
	memset(&response, 0, sizeof(response));
	if(rdmInstance->actHeader.pdl  > 1)
	{
		uint16_t tmp;
		memcpy(&tmp, msg, sizeof(tmp));
		response.requestedSlot = bswap16(tmp);
		RdmExtGetSlotDescription(&implemented, &response);
		if(implemented.response != RDM_NACK_REASON)
		{
			response.requestedSlot = bswap16(response.requestedSlot);
			SendResponse(rdmInstance, (uint8_t *)&response, (uint8_t)(2 + response.labelSize), implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
	(void)msg;
}

static void DecodeGetDefaultSlotValue(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	uint16_t responseSlot[RDMDATALENGTHMAX / 2];
	uint8_t slotCount = 0;
	memset(&responseSlot, 0, sizeof(responseSlot));
	RdmExtGetDefaultSlotValue(&implemented, (rdm_slot_default_get_response_array_t *)&responseSlot, &slotCount);
	if(implemented.response != RDM_NACK_REASON)
	{
		for(uint8_t i = 0; i < slotCount; i++)
		{
			uint16_t slotOffset = ((rdm_slot_default_get_response_array_t *)&responseSlot)[i].slotOffset;
			slotOffset          = bswap16(slotOffset);
			memcpy((uint16_t *)&(((rdm_slot_default_get_response_array_t *)&responseSlot)[i].slotOffset), &slotOffset,
				   2);
		}
		SendResponse(rdmInstance, (uint8_t *)&responseSlot, slotCount * sizeof(rdm_slot_default_get_response_array_t),
					 implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetDisplayInvert(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_display_invert_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetDisplayInvert(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(rdm_display_invert_get_response_t), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetDisplayLevel(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	uint8_t level        = 0;
	RdmExtGetDisplayLevel(&implemented, &level);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&level, 1, implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetPanInvert(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	bool invert          = 0;
	RdmExtGetPanInvert(&implemented, &invert);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&invert, 1, implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetTiltInvert(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	bool invert          = 0;
	RdmExtGetTiltInvert(&implemented, &invert);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&invert, sizeof(invert), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetRealTimeClock(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	uint64_t response    = 0;
	RdmExtGetRealTimeClock(&implemented, &response);

	if(implemented.response != RDM_NACK_REASON)
	{
		response = bswap64(response);
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(uint64_t), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetPresetPlayback(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_preset_playback_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetPresetPlayback(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(rdm_preset_playback_response_t), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;

}
#endif

#ifdef RDM_SUPPORT_DIMMER
static void DecodeGetDimmerInfo(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_dimmer_info_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetDimmerInfo(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		response.minimumLevelLowerLimit = bswap16(response.minimumLevelLowerLimit);
		response.minimumLevelUpperLimit = bswap16(response.minimumLevelUpperLimit);
		response.maximumLevelLowerLimit = bswap16(response.maximumLevelLowerLimit);
		response.maximumLevelUpperLimit = bswap16(response.maximumLevelUpperLimit);
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(rdm_dimmer_info_response_t), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetMinimumLevel(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_minimum_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetMinimumLevel(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		response.minimumLevelIncreasing = bswap16(response.minimumLevelIncreasing);
		response.minimumLevelDecreasing = bswap16(response.minimumLevelDecreasing);
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(rdm_minimum_response_t), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetMaximumLevel(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response          = RDM_ACK;
	implemented.reason            = NR_UNKNOWN_PID;
	uint16_t responseMaximumLevel = 0;
	RdmExtGetMaximumLevel(&implemented, &responseMaximumLevel);
	if(implemented.response != RDM_NACK_REASON)
	{
		responseMaximumLevel = bswap16(responseMaximumLevel);
		SendResponse(rdmInstance, (uint8_t *)&responseMaximumLevel, sizeof(responseMaximumLevel), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetCurve(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_curve_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetCurve(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(rdm_curve_response_t), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetCurveDescription(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl >= 1)
	{
		rdm_curve_description_response_t response;
		memset(&response, 0, sizeof(response));
		response.requestedCurve = msg[0];
		RdmExtGetCurveDescription(&implemented, &response);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, (uint8_t *)&response, 1 + response.labelSize, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
	(void)msg;
}

static void DecodeGetOutputResponseTime(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_output_response_time_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetOutputResponseTime(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(rdm_output_response_time_response_t), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetOutputResponseTimeDescription(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl >= 1)
	{
		rdm_output_response_time_description_response_t response;
		memset(&response, 0, sizeof(response));
		response.requestedResponseTime = msg[0];
		RdmExtGetOutputResponseTimeDescription(&implemented, &response);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, (uint8_t *)&response, sizeof(1 + response.labelSize), implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}

static void DecodeGetModulationFequency(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_modulation_frequency_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetModulationFrequency(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(rdm_modulation_frequency_response_t), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetModulationFequencyDescription(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl >= 1)
	{
		rdm_modulation_frequency_description_response_t response;
		memset(&response, 0, sizeof(response));
		response.requestedModulationSetting = msg[0];
		RdmExtGetModulationFrequencyDescription(&implemented, &response);
		if(implemented.response != RDM_NACK_REASON)
		{
			response.frequency = bswap32(response.frequency);
			SendResponse(rdmInstance, (uint8_t *)&response, sizeof(5 + response.labelSize), implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}
#endif

#if defined(RDM_SUPPORT_QUEUED_MESSAGES) || defined(RDM_SUPPORT_PROXY)
static void DecodeGetQueuedMessage(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl >= 1)
	{
		if(rdmInstance->rdm_queue_handler != NULL && rdmInstance->rdm_queue_count_handler != NULL)
		{
			uint8_t queueCount = 0;
			rdmInstance->rdm_queue_count_handler(rdmInstance->actHeader.targetRdmId, &queueCount, rdmInstance->actId);
			if(queueCount == 0)
			{
#ifdef RTTLOGGINGSET
				LOGGER_INFO("Queue Count 0. No message");
#endif
				// Respond with a status message pdl 0
				rdmInstance->actHeader.pid_num = RDMPID_STATUS_MESSAGES;
				SendResponse(rdmInstance, NULL, 0, implemented.response);
				return;
			}
			uint8_t buf[RDMDATALENGTHMAX];
			uint16_t queuelength = 0;
			// switch(msg[0])
			//{
			//   case RDM_STATUS_GET_LAST_MESSAGE:

			//  break;
			//  default:

			//  break;
			//}
			if(rdmInstance->rdm_queue_handler(rdmInstance->actHeader.targetRdmId, buf, &queuelength, rdmInstance->actId))
			{
				rdmInstance->rdm_answer_handler(buf, queuelength, true, rdmInstance->actId);
			}
		}
		else
		{
			SendNackResponse(rdmInstance, NR_UNKNOWN_PID);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
	(void)msg;
}
#endif

#ifdef RDM_SUPPORT_PROXY
static void DecodeGetProxiedDeviceCount(rdm_library_instance_t *rdmInstance, uint8_t *dataBuffer)
{
  rdm_answer_t implemented;
  implemented.response = RDM_ACK;
  implemented.reason   = NR_UNKNOWN_PID;
   
  rdm_proxy_device_count_response_t response;
  response.deviceCount = rdmInstance->proxyFixtureCount;
  response.listChange = rdmInstance->proxyListUpdated;

  SendResponse(rdmInstance, (uint8_t*)&response, sizeof(response), implemented.response);
  (void)dataBuffer;
}

static void DecodeGetProxiedDevices(rdm_library_instance_t *rdmInstance, uint8_t *dataBuffer)
{
  rdm_answer_t implemented;
  implemented.response = RDM_ACK;
  implemented.reason   = NR_UNKNOWN_PID;
  uint8_t fixtureCount = rdmInstance->proxyFixtureCount;
  if(fixtureCount > 38)
  {
    fixtureCount = 38;
  }
  SendResponse(rdmInstance, (uint8_t*)rdmInstance->proxyrdmIds, 6 * fixtureCount, implemented.response);
  rdmInstance->proxyListUpdated = false;
  (void)dataBuffer;
}

#endif
#ifdef RDM_SUPPORT_CUSTOM_LEDCAL
static void DecodeGetCustomAdjustLedCalibrationData(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	uint8_t buf[sizeof(rdm_adjustledcalibration_get_response_t) +
				sizeof(rdm_adjustledcalibration_colorinfos_get_response_t)];
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	memset(buf, 0, sizeof(buf));
	RdmExtGetCustomAdjustLedCalibrationData(
		&implemented, (rdm_adjustledcalibration_get_response_t *)buf,
		(rdm_adjustledcalibration_colorinfos_get_response_t *)&buf[sizeof(rdm_adjustledcalibration_get_response_t)]);
	if(implemented.response != RDM_NACK_REASON)
	{
		rdm_adjustledcalibration_get_response_t *calResponse = (rdm_adjustledcalibration_get_response_t *)buf;
		rdm_adjustledcalibration_colorinfos_get_response_t *calInfoResponse =
			(rdm_adjustledcalibration_colorinfos_get_response_t *)&buf[sizeof(rdm_adjustledcalibration_get_response_t)];
		calResponse->calibration_compressed_size  = bswap32(calResponse->calibration_compressed_size);
		calResponse->calibration_compressed_crc32 = bswap32(calResponse->calibration_compressed_crc32);
		calResponse->calibration_data_size        = bswap32(calResponse->calibration_data_size);
		calResponse->calibration_data_crc32       = bswap32(calResponse->calibration_data_crc32);
		calResponse->calibration_serial           = bswap64(calResponse->calibration_serial);
		calResponse->calibration_date             = bswap64(calResponse->calibration_date);
		calResponse->calibration_generation_date  = bswap64(calResponse->calibration_generation_date);
		for(uint8_t i = 0; i < calInfoResponse->color_info_count; i++)
		{
			calInfoResponse->color_control_infos[i] = bswap32(calInfoResponse->color_control_infos[i]);
		}
		SendResponse(rdmInstance, buf,
					 (uint8_t)(sizeof(rdm_adjustledcalibration_get_response_t) +
							   calInfoResponse->color_info_count * sizeof(uint32_t)),
					 implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeGetCustomAdjustLedCalibrationDebug(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	rdm_adjustledcalibration_debug_get_response_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetCustomAdjustLedCalibrationDebug(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(response), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}
#endif

#ifdef RDM_SUPPORT_CUSTOM_SERVICEHOURS
static void DecodeGetCustomServiceHours(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	uint32_t response;
	memset(&response, 0, sizeof(response));
	RdmExtGetCustomServiceHours(&implemented, &response);
	if(implemented.response != RDM_NACK_REASON)
	{
		response = bswap32(response);
		SendResponse(rdmInstance, (uint8_t *)&response, sizeof(response), implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}
#endif

#ifdef RDM_SUPPORT_COLORCONVERTER
static void DecodeGetColorConverterLicense(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	uint8_t response[RDMDATALENGTHMAX];
	memset(&response, 0, sizeof(response));
	uint8_t responseSize = 0;
	if(rdmInstance->actHeader.pdl >= 16)
	{
		RdmExtGetColorConverterLicense(&implemented, msg, response, &responseSize);
		if(implemented.response != RDM_NACK_REASON)
		{
			if(responseSize > RDMDATALENGTHMAX)
			{
				responseSize = RDMDATALENGTHMAX;
			}
			SendResponse(rdmInstance, response, responseSize, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
	(void)msg;
}

static void DecodeGetColorConverterCalibration(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	uint8_t response[2];
	memset(&response, 0, sizeof(response));
	if(rdmInstance->actHeader.pdl >= 16)
	{
		RdmExtGetColorConverterCalibration(&implemented, (uint8_t *)response, (uint8_t *)&response[1]);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, response, sizeof(response), implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
	(void)msg;
}

static void DecodeGetColorConverterControl(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	uint8_t response[RDMDATALENGTHMAX];
	memset(&response, 0, sizeof(response));
	uint8_t responseSize = 0;
	if(rdmInstance->actHeader.pdl >= 16)
	{
		RdmExtGetColorConverterControl(&implemented, (uint16_t *)response, &responseSize);
		if(implemented.response != RDM_NACK_REASON)
		{
			if(responseSize > RDMDATALENGTHMAX)
			{
				responseSize = RDMDATALENGTHMAX;
			}
			SendResponse(rdmInstance, response, responseSize, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
	(void)msg;
}
#endif

#ifdef RDM_SUPPORT_CUSTOM_DFU

#define CUSTOM_DFU_COMMAND_STATUS 0x3
static void DecodeGetCustomDfu(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
  rdm_answer_t implemented;
  implemented.response = RDM_ACK;
  implemented.reason   = NR_UNKNOWN_PID;
  uint16_t response[RDMDATALENGTHMAX/2];
  memset(&response, 0, sizeof(response));
  uint8_t responseSize = 0;
  if(rdmInstance->actHeader.pdl >= 4)
  {
    uint8_t protocollVersion = msg[0];
    uint8_t command = msg[1];
    if(command == CUSTOM_DFU_COMMAND_STATUS)
    {
      uint16_t lastPacketIndex = 0;
      memcpy(&lastPacketIndex, &msg[2], sizeof(uint16_t));
      lastPacketIndex = bswap16(lastPacketIndex);
      ((uint8_t*)response)[0] = protocollVersion;
      ((uint8_t*)response)[1] = command;
      RdmExtGetCustomDfuStatus(&implemented,lastPacketIndex, (rdm_custom_response_dfu_status_code_t*)&response[1],&response[2], &responseSize);
      if(implemented.response != RDM_NACK_REASON)
      {
          if(responseSize > RDMDATALENGTHMAX/2)
          {
              responseSize = RDMDATALENGTHMAX/2;
          }
          for(uint8_t i= 0; i< responseSize;i++)
          {
        	  response[2 + i] = bswap16(response[2 + i]);
          }
          SendResponse(rdmInstance, (uint8_t*)response, 4 + responseSize * 2, implemented.response);
      }
      else
      {
		  SendNackResponse(rdmInstance, implemented.reason);
      }
    }
    else
    {
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
    }
  }
  else
  {
	  SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
  }
}
#endif

#ifdef RDM_SUPPORT_REQUIRED
static void DecodeSetIdentify(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl > 0)
	{
		if(msg[0] == 1)
		{
			RdmExtSetIdentify(&implemented, true);
		}
		else
		{
			RdmExtSetIdentify(&implemented, false);
		}

		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}
#endif

#ifdef RDM_SUPPORT_BASIC
static void DecodeSetFactoryDefaults(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	RdmExtSetFactoryDefaults(&implemented);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, NULL, 0, implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
	(void)msg;
}

static void DecodeSetDeviceLabel(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl > RDMLABELLENGTH)
	{
		rdmInstance->actHeader.pdl = RDMLABELLENGTH;
	}
	RdmExtSetDeviceLabel(&implemented, (char *)msg, (uint8_t)rdmInstance->actHeader.pdl);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, NULL, 0, implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
}

static void DecodeSetDmxPersonality(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl > 0)
	{
		RdmExtSetDmxPersonality(&implemented, msg[0]);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}

static void DecodeSetDmxStartaddress(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl > 1)
	{
		uint16_t address = (uint16_t)(msg[1] | (msg[0] << 8));
		RdmExtSetDmxStartaddress(&implemented, address);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}



static void DecodeSetResetDevice(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl > 0)
	{
		RdmExtSetResetDevice(&implemented, msg[0]);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}

static void DecodeSetPowerState(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl > 0)
	{
		RdmExtSetPowerState(&implemented, msg[0]);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}

static void DecodeSetCustomPid(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;

	RdmExtSetCustomPid(&implemented, rdmInstance->actHeader.pid_num, msg, rdmInstance->actHeader.pdl);
	if(implemented.response != RDM_NACK_REASON)
	{
		SendResponse(rdmInstance, NULL, 0, implemented.response);
	}
	else
	{
		SendNackResponse(rdmInstance, implemented.reason);
	}
}
#endif

#ifdef RDM_SUPPORT_SELFTEST
static void DecodeSetPerformSelftest(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl > 0)
	{
		RdmExtSetPerformSelftest(&implemented, msg[0]);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}
#endif

#ifdef RDM_SUPPORT_LAMP
static void DecodeSetLampOnMode(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl >= sizeof(rdm_lamp_on_mode_t))
	{
		RdmExtSetLampOnMode(&implemented, *(rdm_lamp_on_mode_t *)msg);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}

#endif

#ifdef RDM_SUPPORT_EXTENDED
static void DecodeSetLanguage(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl >= sizeof(rdm_language_set_t))
	{
		RdmExtSetLanguage(&implemented, *(rdm_language_set_t *)msg);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}

static void DecodeSetDisplayInvert(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl >= sizeof(rdm_display_invert_t))
	{
		RdmExtSetDisplayInvert(&implemented, *(rdm_display_invert_t *)msg);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}

static void DecodeSetDisplayLevel(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl >= sizeof(uint8_t))
	{
		RdmExtSetDisplayLevel(&implemented, *(uint8_t *)msg);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}

static void DecodeSetPanInvert(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl >= sizeof(bool))
	{
		RdmExtSetPanInvert(&implemented, *(bool *)msg);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}

static void DecodeSetTiltInvert(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl >= sizeof(bool))
	{
		RdmExtSetTiltInvert(&implemented, *(bool *)msg);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}

static void DecodeSetCapturePreset(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl >= sizeof(rdm_capture_preset_set_t))
	{
		RdmExtSetCapturePreset(&implemented, *(rdm_capture_preset_set_t *)msg);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}

static void DecodeSetRealTimeClock(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl >= sizeof(uint64_t))
	{
		uint64_t realtimeclock;
		memcpy(&realtimeclock, msg, sizeof(realtimeclock));
		realtimeclock = bswap64(realtimeclock);
		RdmExtSetRealTimeClock(&implemented, realtimeclock);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}

static void DecodeSetPresetPlayback(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl >= sizeof(rdm_preset_playback_response_t))
	{
		rdm_preset_playback_response_t set = *(rdm_preset_playback_response_t *)msg;
		set.mode                           = bswap16(set.mode);
		RdmExtSetPresetPlayback(&implemented, set);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}
#endif

#ifdef RDM_SUPPORT_DIMMER
static void DecodeSetMinimumLevel(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl >= sizeof(rdm_minimum_response_t))
	{
		rdm_minimum_response_t set = *(rdm_minimum_response_t *)msg;
		set.minimumLevelDecreasing = bswap16(set.minimumLevelDecreasing);
		set.minimumLevelIncreasing = bswap16(set.minimumLevelIncreasing);
		RdmExtSetMinimumLevel(&implemented, set);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}

static void DecodeSetMaximumLevel(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl >= sizeof(rdm_minimum_response_t))
	{
		uint16_t maximum = *(uint16_t *)msg;
		maximum          = bswap16(maximum);
		RdmExtSetMaximumLevel(&implemented, maximum);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}

static void DecodeSetCurve(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl >= sizeof(uint8_t))
	{
		RdmExtSetCurve(&implemented, msg[0]);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}

static void DecodeSetOutputResponseTime(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl >= sizeof(uint8_t))
	{
		RdmExtSetOutputResponseTime(&implemented, msg[0]);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}

static void DecodeSetModulationFequency(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl >= sizeof(uint8_t))
	{
		RdmExtSetModulationFrequency(&implemented, msg[0]);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}
#endif

#ifdef RDM_SUPPORT_CUSTOM_IQ
static void DecodeSetIq(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	uint32_t key         = 0;
	if(rdmInstance->actHeader.pdl >= sizeof(rdm_iq_settings_stream_t) + 1)
	{
		rdm_iq_settings_stream_t *stream = (rdm_iq_settings_stream_t *)msg;
		switch(stream->command)
		{
		case RDMIQSETTINGSCOMMAND_INPUTSOURCE:
			RdmExtSetIqInputSource(&implemented, *((rdm_iq_inputsource_t *)&msg[sizeof(rdm_iq_settings_stream_t)]));
			break;
		case RDMIQSETTINGSCOMMAND_INPUTSTATE:
			RdmExtSetIqInputState(&implemented, *((rdm_iq_set_input_state_t *)&msg[sizeof(rdm_iq_settings_stream_t)]));
			break;
		case RDMIQSETTINGSCOMMAND_DISPLAYPOPUP:
			RdmExtSetIqPopup(&implemented, *((rdm_iq_popup_t *)&msg[sizeof(rdm_iq_settings_stream_t)]));
			break;
		case RDMIQSETTINGSCOMMAND_EMERGENCYMODE:
			RdmExtSetIqEmergencyMode(&implemented, msg[sizeof(rdm_iq_settings_stream_t)] > 0);
			break;
		case RDMIQSETTINGSCOMMAND_BATTERYRUNTIME:
			RdmExtSetIqBatteryRuntime(&implemented,
									  *((rdm_iq_battery_runtime_t *)&msg[sizeof(rdm_iq_settings_stream_t)]));
			break;
		case RDMIQSETTINGSCOMMAND_ANTITHEFTMODE:
			memcpy(&key, &msg[sizeof(rdm_iq_settings_stream_t)], sizeof(uint32_t));
			key = bswap32(key);
			if(key == RDMIQANTITHEFT_KEY_OFF)
			{
				RdmExtSetIqAntitheftMode(&implemented, RDMIQANTITHEFT_OFF);
			}
			else if(key == RDMIQANTITHEFT_KEY_ON)
			{
				RdmExtSetIqAntitheftMode(&implemented, RDMIQANTITHEFT_ON);
			}
			else if(key == RDMIQANTITHEFT_KEY_ALARM)
			{
				RdmExtSetIqAntitheftMode(&implemented, RDMIQANTITHEFT_ALARM);
			}
			else
			{
				implemented.response = RDM_NACK_REASON;
			}
			break;
		default:
			implemented.response = RDM_NACK_REASON;
			break;
		}
		if(implemented.response != RDM_NACK_REASON)
		{
			uint8_t response = 0;
			SendResponse(rdmInstance, &response, 1, implemented.response);
		}
		else
		{
			uint8_t response = 3;  // 0 -Ok , 1 - Unknown Command , 2 Err Execution Command, 3 Unsupported command
			SendResponse(rdmInstance, &response, 1, implemented.response);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}

#endif

#ifdef RDM_SUPPORT_CUSTOM_LEDCAL
static void DecodeSetCustomAdjustLedCalibrationData(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl >= sizeof(rdm_adjustledcalibration_set_t))
	{
		rdm_adjustledcalibration_set_t set;
		memcpy(&set, msg, sizeof(set));
		set.compressed_crc32 = bswap32(set.compressed_crc32);
		set.compressed_size  = bswap32(set.compressed_size);
		set.data_crc32       = bswap32(set.data_crc32);
		set.data_size        = bswap32(set.data_size);
		RdmExtSetCustomAdjustLedCalibrationData(&implemented, set);
		if(implemented.response != RDM_NACK_REASON)
		{
			uint8_t response = 0;
			SendResponse(rdmInstance, &response, 1, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}

static void DecodeSetCustomAdjustLedCalibrationSend(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl >= sizeof(6))
	{
		rdm_adjustledcalibration_send_set_t set;
		memcpy(&set, msg, sizeof(set));
		set.index       = bswap16(set.index);
		set.chunk_crc32 = bswap32(set.chunk_crc32);
		custom_adjust_cal_send_response_t response;
		RdmExtSetCustomAdjustLedCalibrationSend(&implemented, set, &msg[6], (uint8_t)(rdmInstance->actHeader.pdl - 6), &response);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, &response, 1, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}
#endif

#ifdef RDM_SUPPORT_CUSTOM_SERVICEHOURS
static void DecodeSetCustomServiceHours(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl >= sizeof(uint32_t))
	{
		uint32_t set;
		memcpy(&set, msg, sizeof(uint32_t));
		RdmExtSetCustomServiceHours(&implemented, set);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}
#endif

#ifdef RDM_SUPPORT_COLORCONVERTER
static void DecodeSetColorConverterLicense(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl >= sizeof(rdm_colorconverter_license_t))
	{
		rdm_colorconverter_license_t license;
		memcpy(&license, msg, sizeof(license));
		RdmExtSetColorConverterLicense(&implemented, license);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}

static void DecodeSetColorConverterCalibration(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl  > 3 && (rdmInstance->actHeader.pdl  - 3) % 2 == 0)
	{
		uint8_t dataLength  = (rdmInstance->actHeader.pdl  - 3) / 2;
		uint16_t dataOffset = (msg[1] << 8) | msg[2];
		uint16_t data[dataLength];
		memcpy(data, &msg[3], sizeof(data));
		RdmExtSetColorConverterCalibration(&implemented, (rdm_colorconverter_cal_command_t)msg[0], dataOffset, data,
										   dataLength);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}

static void DecodeSetColorConverterControl(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
	rdm_answer_t implemented;
	implemented.response = RDM_ACK;
	implemented.reason   = NR_UNKNOWN_PID;
	if(rdmInstance->actHeader.pdl >= 4)
	{
		RdmExtSetColorConverterControl(&implemented, msg[0], msg[1], (msg[2] << 8) | msg[3]);
		if(implemented.response != RDM_NACK_REASON)
		{
			SendResponse(rdmInstance, NULL, 0, implemented.response);
		}
		else
		{
			SendNackResponse(rdmInstance, implemented.reason);
		}
	}
	else
	{
		SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
	}
}
#endif

#ifdef RDM_SUPPORT_CUSTOM_DFU
#define CUSTOM_DFU_PROTOCOL_VERSION 1
#define CUSTOM_DFU_COMMAND_START 	1
#define CUSTOM_DFU_COMMAND_SEND 	2
#define CUSTOM_DFU_COMMAND_END 		4
static void DecodeSetCustomDfu(rdm_library_instance_t *rdmInstance, uint8_t *msg)
{
  rdm_answer_t implemented;
  implemented.response = RDM_ACK;
  implemented.reason   = NR_UNKNOWN_PID;
  uint8_t response[RDMDATALENGTHMAX];
  memset(&response, 0, sizeof(response));
  uint8_t responseSize = 0;
  if(rdmInstance->actHeader.pdl >= 2)
  {
    uint8_t protocollVersion = msg[0];
    if(protocollVersion != CUSTOM_DFU_PROTOCOL_VERSION)
    {
    	SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
    	return;
    }
    uint8_t command = msg[1];
    switch(command)
    {
    case CUSTOM_DFU_COMMAND_START:
      if(rdmInstance->actHeader.pdl >= 3)
      {
        uint8_t packetLength = msg[2];
        if(rdmInstance->actHeader.pdl >= 3 + packetLength && packetLength < 31)
        {
        	RdmExtSetCustomDfuStart(&implemented,packetLength, &msg[3], (rdm_custom_response_dfu_start_t *)&response[2]);
        	responseSize = sizeof(rdm_custom_response_dfu_start_t);
        	break;
        }
        else
        {
        	SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
        	return;
        }
      }
      SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
      break;
    case CUSTOM_DFU_COMMAND_SEND:
      if(rdmInstance->actHeader.pdl >= 4)
      {
        uint16_t packetIndex = 0;
        memcpy(&packetIndex, &msg[2],sizeof(uint16_t));
        packetIndex = bswap16(packetIndex);
        RdmExtSetCustomDfuSend(&implemented,packetIndex, &msg[4],rdmInstance->actHeader.pdl  - 4);
        return;
      }
    break;
    case CUSTOM_DFU_COMMAND_END:
		RdmExtSetCustomDfuEnd(&implemented,&response[2]);
		responseSize = 1;
    break;
    default:
    	 SendNackResponse(rdmInstance, implemented.reason);
    	 return;
    	break;
    }
    if(implemented.response != RDM_NACK_REASON)
    {
    	response[0] = protocollVersion;
    	response[1] = command;
      SendResponse(rdmInstance, response, responseSize + 2, implemented.response);
    }
    else
    {
      SendNackResponse(rdmInstance, implemented.reason);
    }
  }
  else
  {
    SendNackResponse(rdmInstance, NR_FORMAT_ERROR);
  }
}
#endif


static void SendResponse(rdm_library_instance_t *rdmInstance, uint8_t *data, uint8_t pdl, rdm_response_t rt)
{
	if(rdmInstance->actHeader.isBroadcast || rdmInstance->actHeader.isManufacturerBroadcast)
	{
            // Do not send a response when it is a broadcast message
            return;
	}
	uint16_t msgLength = GetRdmMessageLength(pdl);
	uint8_t responseBuffer[msgLength];
	rdm_message_header_t response;
	memset(&response, 0, sizeof(response));
	if(rdmInstance->actHeader.cc == RDM_GET_COMMAND)
	{
		response.cc = RDM_GET_COMMAND_RESPONSE;
	}
	else if(rdmInstance->actHeader.cc == RDM_SET_COMMAND)
	{
		response.cc = RDM_SET_COMMAND_RESPONSE;
	}
	else if(rdmInstance->actHeader.cc == RDM_DISCOVERY_COMMAND)
	{
		response.cc = RDM_DISCOVERY_COMMAND_RESPONSE;
	}
	response.pdl      = pdl;
	response.pid_num  = rdmInstance->actHeader.pid_num;
	response.response = true;
	response.rt       = rt;
	response.tn       = rdmInstance->actHeader.tn;
#if defined(RDM_SUPPORT_QUEUED_MESSAGES) || defined(RDM_SUPPORT_PROXY)
	if(rdmInstance->rdm_queue_count_handler != NULL)
	{
		rdmInstance->rdm_queue_count_handler(rdmInstance->actHeader.targetRdmId, &(response.msgCount), rdmInstance->actId);
	}
	else
	{
		response.msgCount = 0;
	}
#else
	response.msgCount = 0;
#endif
	memcpy(response.sourceRdmId, rdmInstance->rdm_id, 6);
	memcpy(response.targetRdmId, rdmInstance->actHeader.sourceRdmId, 6);

	EncodeRdmMessage(response, data, responseBuffer);
        if(rdmInstance->rdm_answer_handler != NULL)
        {
          rdmInstance->rdm_answer_handler(responseBuffer, msgLength, true, rdmInstance->actId);
        }
}

static void SendNackResponse(rdm_library_instance_t *rdmInstance, rdm_nack_reasoncodes_t reason)
{
	if(rdmInstance->actHeader.isBroadcast || rdmInstance->actHeader.isManufacturerBroadcast)
	{
		// Do not send a response when it is a broadcast message
		return;
	}
	uint8_t msgLength = (uint8_t)GetRdmMessageLength(2);
	uint8_t responseBuffer[msgLength];
	memset(responseBuffer, 0, msgLength);
	rdm_message_header_t response;
	memset(&response, 0, sizeof(response));
	if(rdmInstance->actHeader.cc == RDM_GET_COMMAND)
	{
		response.cc = RDM_GET_COMMAND_RESPONSE;
	}
	else if(rdmInstance->actHeader.cc == RDM_SET_COMMAND)
	{
		response.cc = RDM_SET_COMMAND_RESPONSE;
	}
	response.pdl      = 2;
	response.pid_num  = rdmInstance->actHeader.pid_num;
	response.response = true;
	response.rt       = RDM_NACK_REASON;
	response.tn       = rdmInstance->actHeader.tn;
#if defined(RDM_SUPPORT_QUEUED_MESSAGES) || defined(RDM_SUPPORT_PROXY)
	if(rdmInstance->rdm_queue_count_handler != NULL)
	{
		rdmInstance->rdm_queue_count_handler(rdmInstance->actHeader.targetRdmId, &(response.msgCount), rdmInstance->actId);
	}
	else
	{
		response.msgCount = 0;
	}
#else
	response.msgCount = 0;
#endif
	memcpy(response.sourceRdmId, rdmInstance->rdm_id, 6);
	memcpy(response.targetRdmId, rdmInstance->actHeader.sourceRdmId, 6);
	uint16_t tempReason = reason;
	tempReason          = bswap16(tempReason);

	EncodeRdmMessage(response, (uint8_t *)&tempReason, responseBuffer);
        if(rdmInstance->rdm_answer_handler != NULL)
        {
          rdmInstance->rdm_answer_handler(responseBuffer, msgLength, true, rdmInstance->actId);
        }
}
#ifdef RDM_SUPPORT_PROXY
static void SendAckTimerResponse(rdm_library_instance_t *rdmInstance, uint16_t timeDelay)
{
	if(rdmInstance->actHeader.isBroadcast || rdmInstance->actHeader.isManufacturerBroadcast)
	{
		// Do not send a response when it is a broadcast message
		return;
	}
	uint8_t msgLength = (uint8_t)GetRdmMessageLength(2);
	uint8_t responseBuffer[msgLength];
	memset(responseBuffer, 0, msgLength);
	rdm_message_header_t response;
	memset(&response, 0, sizeof(response));
	if(rdmInstance->actHeader.cc == RDM_GET_COMMAND)
	{
		response.cc = RDM_GET_COMMAND_RESPONSE;
	}
	else if(rdmInstance->actHeader.cc == RDM_SET_COMMAND)
	{
		response.cc = RDM_SET_COMMAND_RESPONSE;
	}
	response.pdl      = 2;
	response.pid_num  = rdmInstance->actHeader.pid_num;
	response.response = true;
	response.rt       = RDM_ACK_TIMER;
	response.tn       = rdmInstance->actHeader.tn;
#if defined(RDM_SUPPORT_QUEUED_MESSAGES) || defined(RDM_SUPPORT_PROXY)
	if(rdmInstance->rdm_queue_count_handler != NULL)
	{
		rdmInstance->rdm_queue_count_handler(rdmInstance->actHeader.targetRdmId, &(response.msgCount), rdmInstance->actId);
	}
	else
	{
		response.msgCount = 0;
	}
#else
	response.msgCount = 0;
#endif
	memcpy(response.sourceRdmId, rdmInstance->rdm_id, 6);
	memcpy(response.targetRdmId, rdmInstance->actHeader.sourceRdmId, 6);
	uint16_t tempReason = timeDelay;
	tempReason          = bswap16(tempReason);

	EncodeRdmMessage(response, (uint8_t *)&tempReason, responseBuffer);
        if(rdmInstance->rdm_answer_handler != NULL)
        {
          rdmInstance->rdm_answer_handler(responseBuffer, msgLength, true, rdmInstance->actId);
        }
#ifdef RTTLOGGINGSET
	LOGGER_INFO("Send Ack Timer");
#endif
}
#endif
static bool RdmMatch(rdm_library_instance_t *rdmInstance)
{
        #ifdef RDM_SUPPORT_PROXY
          rdmInstance->actIsProxy = false;
        #endif
	if(rdmInstance->actHeader.isBroadcast)
	{

          return true;
	}
	if(rdmInstance->actHeader.isManufacturerBroadcast)
	{
		for(uint8_t i = 0; i < 2; i++)
		{
			if(rdmInstance->actHeader.targetRdmId[i] != rdmInstance->rdm_id[i])
			{
				return false;
			}
		}
		return true;
	}
	for(uint8_t i = 0; i < 6; i++)
	{
		if(rdmInstance->actHeader.targetRdmId[i] != rdmInstance->rdm_id[i])
		{
			return false;
		}
	}
        #ifdef RDM_SUPPORT_PROXY
            rdmInstance->actIsProxy = true;
        #endif
	return true;
}
#ifdef RDM_SUPPORT_PROXY
static void RdmProxyProcess(rdm_library_instance_t *rdmInstance, const uint8_t *msg)
{
	uint8_t index = 0;
	if(RdmProxyMatch(rdmInstance, &index))
	{
		if(rdmInstance->actHeader.cc == RDM_DISCOVERY_COMMAND)
		{
			DecodeRdmProxyDiscovery(rdmInstance, (uint8_t *)&msg[RDMDATAPOSITION], index);
		}
		else if(rdmInstance->actHeader.cc == RDM_GET_COMMAND && rdmInstance->actHeader.pid_num == RDMPID_QUEUED_MESSAGE)
		{
			DecodeGetQueuedMessage(rdmInstance, (uint8_t *)msg);
		}
		else
		{
			// Answer with Ack Timer
			SendAckTimerResponse(rdmInstance, rdmInstance->averageResponseTime100ms);
			// Send Data out
			if(rdmInstance->rdm_proxy_transmit_handler != NULL)
			{
				rdmInstance->rdm_proxy_transmit_handler(msg, rdmInstance->actHeader.pdl + RDMHEADERLENGTH + 2, rdmInstance->actId);
#ifdef RTTLOGGINGSET
				LOGGER_INFO("Transmit Proxy Data");
#endif
			}
		}
	}
}
static bool RdmProxyMatch(rdm_library_instance_t *rdmInstance, uint8_t *index)
{
	for(uint8_t x = 0; x < rdmInstance->proxyFixtureCount; x++)
	{
		bool match = true;
		for(uint8_t i = 0; i < 6; i++)
		{
			if(rdmInstance->actHeader.targetRdmId[i] != rdmInstance->proxyrdmIds[x][i])
			{
				match = false;
				break;
			}
		}
		if(match)
		{
			if(index != NULL)
			{
				*index = x;
			}
			return true;
		}
	}
	return false;
}
#endif

static uint16_t bswap16(uint16_t val)
{
	return ((val & 0xFF) << 8) | (val >> 8);
}

static uint32_t bswap32(uint32_t val)
{
	val = ((val & 0xFF000000) >> 24) | ((val & 0x00FF0000) >> 8) | ((val & 0x0000FF00) << 8) | ((val & 0xFF) << 24);
	return val;
}
static uint64_t bswap64(uint64_t val)
{
	val = ((val << 8) & 0xFF00FF00FF00FF00ULL) | ((val >> 8) & 0x00FF00FF00FF00FFULL);
	val = ((val << 16) & 0xFFFF0000FFFF0000ULL) | ((val >> 16) & 0x0000FFFF0000FFFFULL);
	return (val << 32) | (val >> 32);
}
#ifdef RDM_SUPPORT_REQUIRED
__attribute__((__weak__)) void RdmExtGetDeviceInfo(rdm_answer_t *rdmImp, rdm_device_info_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetIdentify(rdm_answer_t *rdmImp, rdm_identify_device_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetSoftwareVersionLabel(rdm_answer_t *rdmImp,
															 rdm_softwareversion_label_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
#endif
#ifdef RDM_SUPPORT_BASIC
__attribute__((__weak__)) void RdmExtGetFactoryDefaults(rdm_answer_t *rdmImp,
														rdm_factory_default_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}

__attribute__((__weak__)) void RdmExtGetProductDetail(rdm_answer_t *rdmImp,
													  rdm_product_detail_id_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetDeviceModelDescription(rdm_answer_t *rdmImp,
															   rdm_device_model_description_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetManufacturerLabel(rdm_answer_t *rdmImp,
														  rdm_manufacturer_label_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetDeviceLabel(rdm_answer_t *rdmImp, rdm_device_label_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}

__attribute__((__weak__)) void RdmExtGetBootSoftwareVersionLabel(
	rdm_answer_t *rdmImp, rdm_boot_softwareversion_label_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetDmxPersonality(rdm_answer_t *rdmImp,
													   rdm_dmx_personality_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetDmxPersonalityDescription(
	rdm_answer_t *rdmImp, rdm_dmx_personality_description_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetDmxStartaddress(rdm_answer_t *rdmImp,
														rdm_dmx_startaddress_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetSensorDefinition(rdm_answer_t *rdmImp,
														 rdm_sensor_defintion_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetSensorValue(rdm_answer_t *rdmImp, rdm_sensor_value_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetDeviceHours(rdm_answer_t *rdmImp, rdm_device_hours_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetDevicePowerCycles(rdm_answer_t *rdmImp,
														  rdm_device_power_cycle_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}

__attribute__((__weak__)) void RdmExtGetPowerState(rdm_answer_t *rdmImp, rdm_power_state_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}

__attribute__((__weak__)) void RdmExtGetCustomPid(uint16_t pid, uint8_t *msg, uint16_t pdl, rdm_answer_t *rdmImp,
												  uint8_t *responseBuf, uint16_t *responseLength)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
	(void)responseLength;
	(void)pid;
	(void)msg;
	(void)pdl;
}
__attribute__((__weak__)) void RdmExtGetSupportedParameter(rdm_answer_t *rdmImp, uint16_t *responseBuf,
														   uint8_t *pidCount)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
	(void)pidCount;
}
__attribute__((__weak__)) void RdmExtGetParameterDescription(rdm_answer_t *rdmImp,
															 rdm_parameter_description_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetBootSoftwareID(rdm_answer_t *rdmImp, uint32_t *responseBootSoftwareID)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBootSoftwareID;
}
__attribute__((__weak__)) void RdmExtGetLampHours(rdm_answer_t *rdmImp, rdm_lamphours_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
#endif
#ifdef RDM_SUPPORT_CUSTOM_IQ
__attribute__((__weak__)) void RdmExtGetIq(rdm_answer_t *rdmImp, rdm_iq_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetCustomError(rdm_answer_t *rdmImp, rdm_custom_label_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetCustomWarning(rdm_answer_t *rdmImp,
													  rdm_custom_label_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
#endif
#ifdef RDM_SUPPORT_LAMP
__attribute__((__weak__)) void RdmExtGetLampStrikes(rdm_answer_t *rdmImp, rdm_lampstrike_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetLampState(rdm_answer_t *rdmImp, rdm_lampstate_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
#endif
#ifdef RDM_SUPPORT_SELFTEST
__attribute__((__weak__)) void RdmExtGetPerformSelftest(rdm_answer_t *rdmImp,
														rdm_perform_selftest_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetSelfTestDescription(rdm_answer_t *rdmImp,
															rdm_selftest_description_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
#endif
#ifdef RDM_SUPPORT_DIMMER
__attribute__((__weak__)) void RdmExtGetDimmerInfo(rdm_answer_t *rdmImp, rdm_dimmer_info_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetMinimumLevel(rdm_answer_t *rdmImp, rdm_minimum_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetMaximumLevel(rdm_answer_t *rdmImp, uint16_t *responseMaximumLevel)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseMaximumLevel;
}
__attribute__((__weak__)) void RdmExtGetCurve(rdm_answer_t *rdmImp, rdm_curve_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetCurveDescription(rdm_answer_t *rdmImp,
														 rdm_curve_description_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetOutputResponseTime(rdm_answer_t *rdmImp,
														   rdm_output_response_time_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetOutputResponseTimeDescription(
	rdm_answer_t *rdmImp, rdm_output_response_time_description_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetModulationFrequency(rdm_answer_t *rdmImp,
															rdm_modulation_frequency_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetModulationFrequencyDescription(
	rdm_answer_t *rdmImp, rdm_modulation_frequency_description_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
#endif

#ifdef RDM_SUPPORT_EXTENDED
__attribute__((__weak__)) void RdmExtGetLanguageCapabilities(rdm_answer_t *rdmImp, uint8_t *responseBuf,
															 uint8_t *languageCounter)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
	(void)languageCounter;
}
__attribute__((__weak__)) void RdmExtGetLanguage(rdm_answer_t *rdmImp, rdm_language_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetSlotInfo(rdm_answer_t *rdmImp, rdm_slot_info_get_response_array_t *responseBuf,
												 uint8_t *slotCount)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
	(void)slotCount;
}
__attribute__((__weak__)) void RdmExtGetSlotDescription(rdm_answer_t *rdmImp, rdm_slot_desc_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetDefaultSlotValue(rdm_answer_t *rdmImp,
														 rdm_slot_default_get_response_array_t *responseBuf,
														 uint8_t *slotCount)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
	(void)slotCount;
}
__attribute__((__weak__)) void RdmExtGetLampOnMode(rdm_answer_t *rdmImp, rdm_lamp_on_mode_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetDisplayInvert(rdm_answer_t *rdmImp,
													  rdm_display_invert_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetDisplayLevel(rdm_answer_t *rdmImp, uint8_t *level)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)level;
}
__attribute__((__weak__)) void RdmExtGetPanInvert(rdm_answer_t *rdmImp, bool *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetTiltInvert(rdm_answer_t *rdmImp, bool *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetRealTimeClock(rdm_answer_t *rdmImp, uint64_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtGetPresetPlayback(rdm_answer_t *rdmImp,
													   rdm_preset_playback_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
#endif

#ifdef RDM_SUPPORT_CUSTOM_LEDCAL
__attribute__((__weak__)) void RdmExtGetCustomAdjustLedCalibrationData(
	rdm_answer_t *rdmImp, rdm_adjustledcalibration_get_response_t *responseBuf,
	rdm_adjustledcalibration_colorinfos_get_response_t *responseBufColorInfos)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
	(void)responseBufColorInfos;
}
__attribute__((__weak__)) void RdmExtGetCustomAdjustLedCalibrationDebug(
	rdm_answer_t *rdmImp, rdm_adjustledcalibration_debug_get_response_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
#endif
#ifdef RDM_SUPPORT_CUSTOM_SERVICEHOURS
__attribute__((__weak__)) void RdmExtGetCustomServiceHours(rdm_answer_t *rdmImp, uint32_t *responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
#endif

#ifdef RDM_SUPPORT_COLORCONVERTER
__attribute__((__weak__)) void RdmExtGetColorConverterLicense(rdm_answer_t *rdmImp, uint8_t *iv, uint8_t *responseBuf,
															  uint8_t *responseLength)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason   = NR_UNKNOWN_PID;
	(void)responseBuf;
	(void)responseLength;
	(void)iv;
}
__attribute__((__weak__)) void RdmExtGetColorConverterCalibration(rdm_answer_t *rdmImp, uint8_t *responseLedCount,
																  uint8_t *responseDieCount)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason   = NR_UNKNOWN_PID;
	(void)responseLedCount;
	(void)responseDieCount;
}
__attribute__((__weak__)) void RdmExtGetColorConverterControl(rdm_answer_t *rdmImp, uint16_t *responseBuf,
															  uint8_t *length)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason   = NR_UNKNOWN_PID;
	(void)responseBuf;
	(void)length;
}

#endif

#ifdef RDM_SUPPORT_CUSTOM_DFU
__attribute__((__weak__)) void RdmExtGetCustomDfuStatus(rdm_answer_t *rdmImp, uint16_t lastPacketIndex, rdm_custom_response_dfu_status_code_t *responseStatus, uint16_t *responseBuf, uint8_t *responseLength)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseStatus;
}
#endif

__attribute__((__weak__)) void RdmExtSetIdentify(rdm_answer_t *rdmImp, bool set)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)set;
}

#ifdef RDM_SUPPORT_BASIC
__attribute__((__weak__)) void RdmExtSetFactoryDefaults(rdm_answer_t *rdmImp)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
}
__attribute__((__weak__)) void RdmExtSetDeviceLabel(rdm_answer_t *rdmImp, char *label, uint8_t labelLength)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)label;
	(void)labelLength;
}
__attribute__((__weak__)) void RdmExtSetDmxPersonality(rdm_answer_t *rdmImp, uint8_t personality)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)personality;
}
__attribute__((__weak__)) void RdmExtSetDmxStartaddress(rdm_answer_t *rdmImp, uint16_t dmxStartAddress)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)dmxStartAddress;
}

__attribute__((__weak__)) void RdmExtSetResetDevice(rdm_answer_t *rdmImp, rdm_reset_t reset)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)reset;
}
__attribute__((__weak__)) void RdmExtSetPowerState(rdm_answer_t *rdmImp, rdm_powerstate_t powerState)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)powerState;
}
__attribute__((__weak__)) void RdmExtSetCustomPid(rdm_answer_t *rdmImp, uint16_t pid, uint8_t *data, uint16_t pdl)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)pid;
	(void)data;
	(void)pdl;
}
#endif

#ifdef RDM_SUPPORT_CUSTOM_IQ
__attribute__((__weak__)) void RdmExtSetIqInputSource(rdm_answer_t *rdmImp, rdm_iq_inputsource_t inputSource)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)inputSource;
}
__attribute__((__weak__)) void RdmExtSetIqInputState(rdm_answer_t *rdmImp, rdm_iq_set_input_state_t inputState)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)inputState;
}
__attribute__((__weak__)) void RdmExtSetIqPopup(rdm_answer_t *rdmImp, rdm_iq_popup_t popup)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)popup;
}
__attribute__((__weak__)) void RdmExtSetIqEmergencyMode(rdm_answer_t *rdmImp, bool emergencyOnOff)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)emergencyOnOff;
}
__attribute__((__weak__)) void RdmExtSetIqBatteryRuntime(rdm_answer_t *rdmImp, rdm_iq_battery_runtime_t runTime)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)runTime;
}
__attribute__((__weak__)) void RdmExtSetIqAntitheftMode(rdm_answer_t *rdmImp, rdm_iq_antitheft_t antitheft)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)antitheft;
}
#endif

#ifdef RDM_SUPPORT_LAMP
__attribute__((__weak__)) void RdmExtSetLampOnMode(rdm_answer_t *rdmImp, rdm_lamp_on_mode_t lamp_mode)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)lamp_mode;
}
#endif

#ifdef RDM_SUPPORT_SELFTEST
__attribute__((__weak__)) void RdmExtSetPerformSelftest(rdm_answer_t *rdmImp, uint8_t selftest)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)selftest;
}
#endif

#ifdef RDM_SUPPORT_DIMMER
__attribute__((__weak__)) void RdmExtSetMinimumLevel(rdm_answer_t *rdmImp, rdm_minimum_response_t minimumLevel)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)minimumLevel;
}
__attribute__((__weak__)) void RdmExtSetMaximumLevel(rdm_answer_t *rdmImp, uint16_t maximumLevel)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)maximumLevel;
}
__attribute__((__weak__)) void RdmExtSetCurve(rdm_answer_t *rdmImp, uint8_t curve)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)curve;
}
__attribute__((__weak__)) void RdmExtSetOutputResponseTime(rdm_answer_t *rdmImp, uint8_t responseTime)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseTime;
}
__attribute__((__weak__)) void RdmExtSetModulationFrequency(rdm_answer_t *rdmImp, uint8_t modulationFrequency)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)modulationFrequency;
}

#endif

#ifdef RDM_SUPPORT_EXTENDED
__attribute__((__weak__)) void RdmExtSetLanguage(rdm_answer_t *rdmImp, rdm_language_set_t language)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)language;
}
__attribute__((__weak__)) void RdmExtSetDisplayInvert(rdm_answer_t *rdmImp, rdm_display_invert_t displayInvert)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)displayInvert;
}
__attribute__((__weak__)) void RdmExtSetDisplayLevel(rdm_answer_t *rdmImp, uint8_t displayLevel)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)displayLevel;
}
__attribute__((__weak__)) void RdmExtSetPanInvert(rdm_answer_t *rdmImp, bool invert)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)invert;
}
__attribute__((__weak__)) void RdmExtSetTiltInvert(rdm_answer_t *rdmImp, bool invert)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)invert;
}
__attribute__((__weak__)) void RdmExtSetCapturePreset(rdm_answer_t *rdmImp, rdm_capture_preset_set_t set)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)set;
}
__attribute__((__weak__)) void RdmExtSetRealTimeClock(rdm_answer_t *rdmImp, uint64_t rtc)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)rtc;
}
__attribute__((__weak__)) void RdmExtSetPresetPlayback(rdm_answer_t *rdmImp, rdm_preset_playback_response_t swap)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)swap;
}
#endif

#ifdef RDM_SUPPORT_CUSTOM_LEDCAL
__attribute__((__weak__)) void RdmExtSetCustomAdjustLedCalibrationData(rdm_answer_t *rdmImp,
																	   rdm_adjustledcalibration_set_t responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)responseBuf;
}
__attribute__((__weak__)) void RdmExtSetCustomAdjustLedCalibrationSend(rdm_answer_t *rdmImp,
																	   rdm_adjustledcalibration_send_set_t set,
																	   uint8_t *data, uint8_t chunkSize,
																	   custom_adjust_cal_send_response_t *response)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNKNOWN_PID;
	(void)set;
	(void)data;
	(void)chunkSize;
	(void)response;
}
#endif

#ifdef RDM_SUPPORT_CUSTOM_SERVICEHOURS
__attribute__((__weak__)) void RdmExtSetCustomServiceHours(rdm_answer_t *rdmImp, uint32_t counter)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason   = NR_UNKNOWN_PID;
	(void)counter;
}
#endif

#ifdef RDM_SUPPORT_COLORCONVERTER
__attribute__((__weak__)) void RdmExtSetColorConverterLicense(rdm_answer_t *rdmImp, rdm_colorconverter_license_t key)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason   = NR_UNKNOWN_PID;
	(void)key;
}
__attribute__((__weak__)) void RdmExtSetColorConverterCalibration(rdm_answer_t *rdmImp,
																  rdm_colorconverter_cal_command_t command,
																  uint16_t dataOffset, uint16_t *calData,
																  uint8_t length)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason   = NR_UNKNOWN_PID;
	(void)command;
	(void)dataOffset;
	(void)calData;
	(void)length;
}
__attribute__((__weak__)) void RdmExtSetColorConverterControl(rdm_answer_t *rdmImp, uint8_t ledIndex, uint8_t dieIndex,
															  uint16_t dimmerValue)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason   = NR_UNKNOWN_PID;
	(void)ledIndex;
	(void)dieIndex;
	(void)dimmerValue;
}
#endif

#ifdef RDM_SUPPORT_CUSTOM_DFU
__attribute__((__weak__)) void RdmExtSetCustomDfuStart(rdm_answer_t *rdmImp, uint8_t dfuPacket8XLength,const uint8_t *startData, rdm_custom_response_dfu_start_t *response)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason   = NR_UNKNOWN_PID;
	(void)dfuPacket8XLength;
	(void)startData;
	(void)response;
}

__attribute__((__weak__)) void  RdmExtSetCustomDfuSend(rdm_answer_t *rdmImp, uint16_t packetIndex, uint8_t *data, uint8_t length)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason   = NR_UNKNOWN_PID;
	(void)packetIndex;
	(void)data;
	(void)length;
}

__attribute__((__weak__)) void RdmExtSetCustomDfuEnd(rdm_answer_t *rdmImp, rdm_custom_response_dfu_status_code_t *response)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason   = NR_UNKNOWN_PID;
	(void)response;
}
#endif
