#ifndef RDMHANDLER_H__
#define RDMHANDLER_H__

#include <stdbool.h>
#include <stdint.h>

#include "rdmlib.h"

/**
 * @file rdmhandler.h
 * @brief RDM handler library interface (fixture-side implementation).
 *
 * This header defines the public API of the RDM handler and the callback
 * interface that the fixture firmware must implement.
 *
 * The library parses RDM requests and calls user callbacks (RdmExtGet* /RdmExtSet*)
 * to fill response payloads or to execute actions.
 *
 * @par Thread-safety
 * This library is NOT thread-safe.
 * - Do NOT call @ref rdm_decodeMessage concurrently (no parallel calls).
 * - If you use interrupts/RTOS, make sure calls are serialized.
 *
 * @par Buffer ownership and lifetime
 * All `responseBuf` pointers passed into callbacks are provided by the library.
 * - The callback must only WRITE response data into the provided buffer.
 * - The buffer is only valid during the callback call. Do not store the pointer.
 *
 * @par Error handling
 * Each callback receives an @ref rdm_answer_t object.
 * - Default is SUCCESS (no change needed for normal behavior).
 * - Only set `rdmImp->response` and `rdmImp->reason` in error cases.
 *
 * @par Feature selection
 * The user defines `RDM_SUPPORT_*` macros to reduce code size and enable
 * only required RDM features and PIDs.
 *
 * @par Supported Parameters (IMPORTANT)
 * Supported PIDs are reported via @ref RdmExtGetSupportedParameter.
 * The user must manually add the PIDs that are implemented/enabled.
 */
 
 /**
 * @brief Callback return state for one RDM request handling.
 *
 * `response` indicates the RDM result (ACK/NACK/...) and
 * `reason` contains the NACK reason code if needed.
 *
 * @note Default behavior is success. The library initializes this structure
 *       as "success". The callback only needs to change it on errors.
 */
typedef struct
{
	rdm_response_t response;
	rdm_nack_reasoncodes_t reason;
} rdm_answer_t;

#ifdef RDM_SUPPORT_MINIMAL
/** @brief Minimal feature set: Discovery + Required + Custom IQ. */
#define RDM_SUPPORT_DISCOVERY
#define RDM_SUPPORT_REQUIRED
#define RDM_SUPPORT_CUSTOM_IQ
#endif

#ifdef RDM_SUPPORT_IQ
/** @brief IQ feature set: Required + Custom IQ + Basic. */
#define RDM_SUPPORT_REQUIRED
#define RDM_SUPPORT_CUSTOM_IQ
#define RDM_SUPPORT_BASIC
#endif

#ifdef RDM_SUPPORT_IQ_DFU
/** @brief IQ + DFU feature set: Discovery + Required + Custom DFU + Custom IQ. */
#define RDM_SUPPORT_DISCOVERY
#define RDM_SUPPORT_REQUIRED
#define RDM_SUPPORT_CUSTOM_DFU
#define RDM_SUPPORT_CUSTOM_IQ
#endif

#ifdef RDM_SUPPORT_FULL
/** @brief Full feature set (largest code size). */
#define RDM_SUPPORT_CUSTOM_SERVICEHOURS
#define RDM_SUPPORT_CUSTOM_LEDCAL
#define RDM_SUPPORT_EXTENDED
#define RDM_SUPPORT_DIMMER
#define RDM_SUPPORT_SELFTEST
#define RDM_SUPPORT_LAMP
#define RDM_SUPPORT_DISCOVERY
#define RDM_SUPPORT_CUSTOM_IQ
#define RDM_SUPPORT_BASIC
#define RDM_SUPPORT_REQUIRED
#define RDM_SUPPORT_CUSTOM_DFU
#endif

/** @brief Maximum RDM payload data length supported by this library. */
#define RDMDATALENGTHMAX 230

/** @brief Maximum slot-info array elements that fit into one RDM payload. */
#define RDM_MAX_SLOTINFO (RDMDATALENGTHMAX / sizeof(rdm_slot_info_get_response_array_t))

#ifdef RDM_SUPPORT_PROXY
/**
 * @brief Maximum number of proxied fixtures.
 * @note Proxy mode is currently not documented in detail (by request).
 */
#define MAX_PROXY_FIXTURES 100
#endif

/**
 * @brief RDM library instance data (must be stored by the application).
 *
 * The application creates an instance of this struct and passes it to the
 * library functions.
 *
 * @note Do not modify internal state fields directly at runtime unless the API
 *       explicitly tells you. Use the provided setter functions.
 */
typedef struct
{
  bool initialized;               /**< True after @ref rdm_init was called successfully. */
  bool discoveryMute;             /**< Discovery mute state (internal). */
  uint8_t rdm_id[6];              /**< Fixture RDM UID (6 bytes: ManufacturerID + DeviceID). */
  rdm_message_header_t actHeader; /**< Current active request header (internal). */
  uint8_t actId;                  /**< Current internal context id (internal). */

  #ifdef RDM_SUPPORT_PROXY
  bool actIsProxy;
  uint8_t proxyrdmIds[MAX_PROXY_FIXTURES][6];
  bool proxyDiscoveryMute[MAX_PROXY_FIXTURES];
  uint8_t proxyFixtureCount;
  uint16_t averageResponseTime100ms;
  bool proxyListUpdated;
  #endif

  // Handlers / callbacks
  #ifdef RDM_SUPPORT_PROXY
  /**
   * @brief Proxy transmit callback.
   * @note Proxy mode is currently not documented in detail (by request).
   */
  void (*rdm_proxy_transmit_handler)(const uint8_t *data, uint16_t length, uint8_t _id);
  #endif

  /**
   * @brief Raw answer output callback used by the library.
   *
   * The library calls this function when it has built an RDM response message
   * and it should be transmitted on the RDM line.
   *
   * @param answer     Pointer to full RDM message (including header and checksum).
   * @param length     Length in bytes.
   * @param sentBreak  True if a BREAK was sent before the response (depending on your transport).
   * @param _id        Context id (used for internal/proxy routing; for normal fixtures you can ignore).
   *
   * @note `answer` is only valid during this call.
   */
  void (*rdm_answer_handler)(const uint8_t *answer, uint16_t length, bool sentBreak, uint8_t _id);

  #if defined(RDM_SUPPORT_QUEUED_MESSAGES) || defined(RDM_SUPPORT_PROXY)
  /**
   * @brief Callback to fetch queued RDM response data.
   *
   * Used if the application maintains a response queue.
   *
   * @param rdmId   RDM UID (6 bytes) of the target fixture.
   * @param data    Output buffer for queued message.
   * @param length  In/out length. Set output length when returning true.
   * @param _id     Context id (internal/proxy).
   *
   * @return true if a queued message was provided; false if none available.
   */
  bool (*rdm_queue_handler)(uint8_t *rdmId, uint8_t *data, uint16_t *length, uint8_t _id);

  /**
   * @brief Callback to report the current queue count.
   *
   * @param rdmId               RDM UID (6 bytes) of the target fixture.
   * @param responseQueueCount  Output queue count.
   * @param _id                 Context id (internal/proxy).
   */
  void (*rdm_queue_count_handler)(uint8_t *rdmId, uint8_t *responseQueueCount, uint8_t _id);
  #endif
} rdm_library_instance_t;

/**
 * @brief Initialize the RDM handler.
 *
 * @param rmdInstance        Pointer to the instance struct (owned by application).
 * @param _rdm_id            6-byte RDM UID of this fixture (must stay valid during call only).
 * @param rdm_answer_callback Callback that transmits the generated RDM response.
 *
 * @return true on success, false otherwise.
 *
 * @note Not thread-safe. Call once during system init.
 */
bool rdm_init(rdm_library_instance_t * rmdInstance, const uint8_t *_rdm_id,
			  void (*rdm_answer_callback)(const uint8_t *msg, uint16_t length, bool sentBreak, uint8_t _id));

/**
 * @brief Update the fixture RDM UID.
 *
 * @param rmdInstance Pointer to instance.
 * @param _rdm_id     6-byte RDM UID.
 *
 * @note Must not be called concurrently with @ref rdm_decodeMessage.
 */
void rdm_setRdmId(rdm_library_instance_t * rmdInstance, const uint8_t *_rdm_id);

/**
 * @brief Decode and handle one incoming RDM message.
 *
 * The library parses the request, then calls the required `RdmExtGet*` / `RdmExtSet*`
 * callbacks (implemented by the application) to build the response.
 *
 * @param rmdInstance Pointer to instance.
 * @param msg         Incoming RDM packet.
 * @param length      Packet length.
 * @param _id         Context id (for normal fixtures: pass 0).
 *
 * @warning Not thread-safe. Do NOT call this function in parallel.
 */
void rdm_decodeMessage(rdm_library_instance_t * rmdInstance, const uint8_t *msg, uint16_t length, uint8_t _id);

#ifdef RDM_SUPPORT_PROXY
/**
 * @brief Configure proxied RDM UIDs (proxy mode).
 * @note Proxy mode is currently not documented in detail (by request).
 */
void rdm_setProxyRdmIds(rdm_library_instance_t * rmdInstance, const uint8_t *_rdm_ids, uint8_t length, uint16_t _averageResponseTime100ms,
						void (*rdm_proxy_transmit_callback)(const uint8_t *data, uint16_t length, uint8_t _id));
#endif

#if defined(RDM_SUPPORT_QUEUED_MESSAGES) || defined(RDM_SUPPORT_PROXY)
/**
 * @brief Register callbacks for queued messages.
 *
 * @param rmdInstance Pointer to instance.
 * @param rdm_queue_count_callback Callback to report queue count.
 * @param rdm_proxy_queue_callback Callback to fetch queued message content.
 */
void rdm_registerQueueCallbacks(rdm_library_instance_t * rmdInstance, void (*rdm_queue_count_callback)(uint8_t *rdmId, uint8_t *responseQueueCount,
																 uint8_t _id),
								bool (*rdm_proxy_queue_callback)(uint8_t *rdmId, uint8_t *data, uint16_t *length,
																 uint8_t _id));
#endif

/* -------------------------------------------------------------------------- */
/*                       RDM callbacks: REQUIRED feature                       */
/* -------------------------------------------------------------------------- */

#ifdef RDM_SUPPORT_REQUIRED

/**
 * @brief GET DEVICE_INFO callback (ESTA RDM).
 *
 * The library calls this when it must answer PID DEVICE_INFO.
 *
 * @param rdmImp       In/out answer status. Default is success; set only on error.
 * @param responseBuf  Buffer to fill with DEVICE_INFO response data.
 *
 * @note `responseBuf` is provided by the library and valid only during this call.
 */
void RdmExtGetDeviceInfo(rdm_answer_t *rdmImp, rdm_device_info_get_response_t *responseBuf);

/** @brief GET SOFTWARE_VERSION_LABEL callback (ESTA RDM). */
void RdmExtGetSoftwareVersionLabel(rdm_answer_t *rdmImp, rdm_softwareversion_label_get_response_t *responseBuf);

/** @brief GET IDENTIFY_DEVICE callback (ESTA RDM). */
void RdmExtGetIdentify(rdm_answer_t *rdmImp, rdm_identify_device_get_response_t *responseBuf);

#endif

/* -------------------------------------------------------------------------- */
/*                 RDM callbacks: BASIC and/or CUSTOM_DFU feature              */
/* -------------------------------------------------------------------------- */

#if defined(RDM_SUPPORT_BASIC) || defined(RDM_SUPPORT_CUSTOM_DFU)

/** @brief GET DEVICE_MODEL_DESCRIPTION callback (ESTA RDM). */
void RdmExtGetDeviceModelDescription(rdm_answer_t *rdmImp, rdm_device_model_description_get_response_t *responseBuf);

/** @brief GET MANUFACTURER_LABEL callback (ESTA RDM). */
void RdmExtGetManufacturerLabel(rdm_answer_t *rdmImp, rdm_manufacturer_label_get_response_t *responseBuf);

/** @brief GET BOOT_SOFTWARE_VERSION_LABEL callback (ESTA RDM). */
void RdmExtGetBootSoftwareVersionLabel(rdm_answer_t *rdmImp,
									   rdm_boot_softwareversion_label_get_response_t *responseBuf);
#endif

/* -------------------------------------------------------------------------- */
/*                         RDM callbacks: BASIC feature                         */
/* -------------------------------------------------------------------------- */

#ifdef RDM_SUPPORT_BASIC

/** @brief GET FACTORY_DEFAULTS callback (ESTA RDM). */
void RdmExtGetFactoryDefaults(rdm_answer_t *rdmImp, rdm_factory_default_get_response_t *responseBuf);

/** @brief GET PRODUCT_DETAIL_ID_LIST callback (ESTA RDM). */
void RdmExtGetProductDetail(rdm_answer_t *rdmImp, rdm_product_detail_id_get_response_t *responseBuf);

/** @brief GET DEVICE_LABEL callback (ESTA RDM). */
void RdmExtGetDeviceLabel(rdm_answer_t *rdmImp, rdm_device_label_get_response_t *responseBuf);

/** @brief GET DMX_PERSONALITY callback (ESTA RDM). */
void RdmExtGetDmxPersonality(rdm_answer_t *rdmImp, rdm_dmx_personality_get_response_t *responseBuf);

/** @brief GET DMX_PERSONALITY_DESCRIPTION callback (ESTA RDM). */
void RdmExtGetDmxPersonalityDescription(rdm_answer_t *rdmImp,
										rdm_dmx_personality_description_get_response_t *responseBuf);

/** @brief GET DMX_START_ADDRESS callback (ESTA RDM). */
void RdmExtGetDmxStartaddress(rdm_answer_t *rdmImp, rdm_dmx_startaddress_get_response_t *responseBuf);

/**
 * @brief GET SENSOR_DEFINITION callback (ESTA RDM).
 *
 * @note The function name contains a spelling issue (`Defintion`) and is kept
 *       for compatibility.
 */
void RdmExtGetSensorDefinition(rdm_answer_t *rdmImp, rdm_sensor_defintion_get_response_t *responseBuf);

/** @brief GET SENSOR_VALUE callback (ESTA RDM). */
void RdmExtGetSensorValue(rdm_answer_t *rdmImp, rdm_sensor_value_get_response_t *responseBuf);

/** @brief GET DEVICE_HOURS callback (ESTA RDM). */
void RdmExtGetDeviceHours(rdm_answer_t *rdmImp, rdm_device_hours_get_response_t *responseBuf);

/** @brief GET LAMP_HOURS callback (ESTA RDM). */
void RdmExtGetLampHours(rdm_answer_t *rdmImp, rdm_lamphours_get_response_t *responseBuf);

/** @brief GET LAMP_STRIKES callback (ESTA RDM). */
void RdmExtGetLampStrikes(rdm_answer_t *rdmImp, rdm_lampstrike_get_response_t *responseBuf);

/** @brief GET LAMP_STATE callback (ESTA RDM). */
void RdmExtGetLampState(rdm_answer_t *rdmImp, rdm_lampstate_get_response_t *responseBuf);

/** @brief GET DEVICE_POWER_CYCLES callback (ESTA RDM). */
void RdmExtGetDevicePowerCycles(rdm_answer_t *rdmImp, rdm_device_power_cycle_get_response_t *responseBuf);

/** @brief GET POWER_STATE callback (ESTA RDM). */
void RdmExtGetPowerState(rdm_answer_t *rdmImp, rdm_power_state_get_response_t *responseBuf);

/**
 * @brief GET SUPPORTED_PARAMETERS callback (ESTA RDM).
 *
 * The application must fill the list of supported PIDs manually.
 * This list should match the enabled features and implemented callbacks.
 *
 * @param rdmImp     In/out answer status (set only on error).
 * @param responseBuf Output array for PIDs.
 * @param pidCount   Output: number of PIDs written.
 *
 * @note The library provides the buffer and it is only valid during this call.
 */
void RdmExtGetSupportedParameter(rdm_answer_t *rdmImp, uint16_t *responseBuf, uint8_t *pidCount);

/** @brief GET PARAMETER_DESCRIPTION callback (ESTA RDM). */
void RdmExtGetParameterDescription(rdm_answer_t *rdmImp, rdm_parameter_description_get_response_t *responseBuf);

/**
 * @brief GET custom PID callback (manufacturer specific).
 *
 * This is the generic hook for custom/manufacturer PIDs.
 * Implement your own PID parsing and response generation here.
 *
 * @param pid            Requested PID.
 * @param msg            Pointer to request data (PDL bytes).
 * @param pdl            Request data length.
 * @param rdmImp         In/out answer status (default success).
 * @param responseBuf    Response payload buffer.
 * @param responseLength Output response length in bytes (payload only).
 */
void RdmExtGetCustomPid(uint16_t pid, uint8_t *msg, uint16_t pdl, rdm_answer_t *rdmImp, uint8_t *responseBuf,
						uint16_t *responseLength);

/** @brief GET BOOT_SOFTWARE_ID callback (ESTA RDM). */
void RdmExtGetBootSoftwareID(rdm_answer_t *rdmImp, uint32_t *responseBootSoftwareID);

#endif

/* -------------------------------------------------------------------------- */
/*                       RDM callbacks: CUSTOM IQ feature                      */
/* -------------------------------------------------------------------------- */

#ifdef RDM_SUPPORT_CUSTOM_IQ

/** @brief GET custom IQ data callback (manufacturer specific). */
void RdmExtGetIq(rdm_answer_t *rdmImp, rdm_iq_get_response_t *responseBuf);

/** @brief GET custom error label callback (manufacturer specific). */
void RdmExtGetCustomError(rdm_answer_t *rdmImp, rdm_custom_label_get_response_t *responseBuf);

/** @brief GET custom warning label callback (manufacturer specific). */
void RdmExtGetCustomWarning(rdm_answer_t *rdmImp, rdm_custom_label_get_response_t *responseBuf);

#endif

/* -------------------------------------------------------------------------- */
/*                     RDM callbacks: SELFTEST / LAMP / EXTENDED               */
/* -------------------------------------------------------------------------- */

#ifdef RDM_SUPPORT_SELFTEST
/** @brief GET PERFORM_SELFTEST callback (ESTA RDM). */
void RdmExtGetPerformSelftest(rdm_answer_t *rdmImp, rdm_perform_selftest_get_response_t *responseBuf);

/** @brief GET SELF_TEST_DESCRIPTION callback (ESTA RDM). */
void RdmExtGetSelfTestDescription(rdm_answer_t *rdmImp, rdm_selftest_description_get_response_t *responseBuf);
#endif

#ifdef RDM_SUPPORT_LAMP
/** @brief GET LAMP_ON_MODE callback (ESTA RDM). */
void RdmExtGetLampOnMode(rdm_answer_t *rdmImp, rdm_lamp_on_mode_response_t *responseBuf);
#endif

#ifdef RDM_SUPPORT_EXTENDED
/** @brief GET LANGUAGE_CAPABILITIES callback (ESTA RDM). */
void RdmExtGetLanguageCapabilities(rdm_answer_t *rdmImp, uint8_t *responseBuf, uint8_t *languageCounter);

/** @brief GET LANGUAGE callback (ESTA RDM). */
void RdmExtGetLanguage(rdm_answer_t *rdmImp, rdm_language_get_response_t *responseBuf);

/** @brief GET SLOT_INFO callback (ESTA RDM). */
void RdmExtGetSlotInfo(rdm_answer_t *rdmImp, rdm_slot_info_get_response_array_t *responseBuf, uint8_t *slotCount);

/** @brief GET SLOT_DESCRIPTION callback (ESTA RDM). */
void RdmExtGetSlotDescription(rdm_answer_t *rdmImp, rdm_slot_desc_get_response_t *responseBuf);

/** @brief GET DEFAULT_SLOT_VALUE callback (ESTA RDM). */
void RdmExtGetDefaultSlotValue(rdm_answer_t *rdmImp, rdm_slot_default_get_response_array_t *responseBuf,
							   uint8_t *slotCount);

/** @brief GET DISPLAY_INVERT callback (ESTA RDM). */
void RdmExtGetDisplayInvert(rdm_answer_t *rdmImp, rdm_display_invert_get_response_t *responseBuf);

/** @brief GET DISPLAY_LEVEL callback (ESTA RDM). */
void RdmExtGetDisplayLevel(rdm_answer_t *rdmImp, uint8_t *responseBuf);

/** @brief GET PAN_INVERT callback (ESTA RDM). */
void RdmExtGetPanInvert(rdm_answer_t *rdmImp, bool *responseBuf);

/** @brief GET TILT_INVERT callback (ESTA RDM). */
void RdmExtGetTiltInvert(rdm_answer_t *rdmImp, bool *responseBuf);

/** @brief GET REAL_TIME_CLOCK callback (ESTA RDM). */
void RdmExtGetRealTimeClock(rdm_answer_t *rdmImp, uint64_t *responseBuf);

/** @brief GET PRESET_PLAYBACK callback (ESTA RDM). */
void RdmExtGetPresetPlayback(rdm_answer_t *rdmImp, rdm_preset_playback_response_t *responseBuf);
#endif

/* -------------------------------------------------------------------------- */
/*                          RDM callbacks: DIMMER feature                       */
/* -------------------------------------------------------------------------- */

#ifdef RDM_SUPPORT_DIMMER
/** @brief GET DIMMER_INFO callback (ESTA RDM). */
void RdmExtGetDimmerInfo(rdm_answer_t *rdmImp, rdm_dimmer_info_response_t *responseBuf);

/** @brief GET MINIMUM_LEVEL callback (ESTA RDM). */
void RdmExtGetMinimumLevel(rdm_answer_t *rdmImp, rdm_minimum_response_t *responseBuf);

/** @brief GET MAXIMUM_LEVEL callback (ESTA RDM). */
void RdmExtGetMaximumLevel(rdm_answer_t *rdmImp, uint16_t *responseMaximumLevel);

/** @brief GET CURVE callback (ESTA RDM). */
void RdmExtGetCurve(rdm_answer_t *rdmImp, rdm_curve_response_t *responseBuf);

/** @brief GET CURVE_DESCRIPTION callback (ESTA RDM). */
void RdmExtGetCurveDescription(rdm_answer_t *rdmImp, rdm_curve_description_response_t *responseBuf);

/** @brief GET OUTPUT_RESPONSE_TIME callback (ESTA RDM). */
void RdmExtGetOutputResponseTime(rdm_answer_t *rdmImp, rdm_output_response_time_response_t *responseBuf);

/** @brief GET OUTPUT_RESPONSE_TIME_DESCRIPTION callback (ESTA RDM). */
void RdmExtGetOutputResponseTimeDescription(rdm_answer_t *rdmImp,
											rdm_output_response_time_description_response_t *responseBuf);

/** @brief GET MODULATION_FREQUENCY callback (ESTA RDM). */
void RdmExtGetModulationFrequency(rdm_answer_t *rdmImp, rdm_modulation_frequency_response_t *responseBuf);

/** @brief GET MODULATION_FREQUENCY_DESCRIPTION callback (ESTA RDM). */
void RdmExtGetModulationFrequencyDescription(rdm_answer_t *rdmImp,
											 rdm_modulation_frequency_description_response_t *responseBuf);
#endif

/* -------------------------------------------------------------------------- */
/*                      RDM callbacks: CUSTOM LEDCAL feature                    */
/* -------------------------------------------------------------------------- */

#ifdef RDM_SUPPORT_CUSTOM_LEDCAL
/** @brief GET custom LED calibration data callback (manufacturer specific). */
void RdmExtGetCustomAdjustLedCalibrationData(rdm_answer_t *rdmImp, rdm_adjustledcalibration_get_response_t *responseBuf,
											 rdm_adjustledcalibration_colorinfos_get_response_t *responseBufColorInfos);

/** @brief GET custom LED calibration debug callback (manufacturer specific). */
void RdmExtGetCustomAdjustLedCalibrationDebug(rdm_answer_t *rdmImp,
											  rdm_adjustledcalibration_debug_get_response_t *responseBuf);
#endif

#ifdef RDM_SUPPORT_CUSTOM_SERVICEHOURS
/** @brief GET custom service hours callback (manufacturer specific). */
void RdmExtGetCustomServiceHours(rdm_answer_t *rdmImp, uint32_t *responseBuf);
#endif

#ifdef RDM_SUPPORT_COLORCONVERTER
/** @brief GET color converter license callback (manufacturer specific). */
void RdmExtGetColorConverterLicense(rdm_answer_t *rdmImp, uint8_t *iv, uint8_t *responseBuf, uint8_t *responseLength);

/** @brief GET color converter calibration status callback (manufacturer specific). */
void RdmExtGetColorConverterCalibration(rdm_answer_t *rdmImp, uint8_t *responseLedCount, uint8_t *responseDieCount);

/** @brief GET color converter control state callback (manufacturer specific). */
void RdmExtGetColorConverterControl(rdm_answer_t *rdmImp, uint16_t *responseBuf, uint8_t *length);
#endif

#ifdef RDM_SUPPORT_CUSTOM_DFU
/** @brief GET custom DFU status callback (manufacturer specific). */
void RdmExtGetCustomDfuStatus(rdm_answer_t *rdmImp, uint16_t lastPacketIndex, rdm_custom_response_dfu_status_code_t *responseStatus, uint16_t *responseBuf, uint8_t *responseLength);
#endif

/* -------------------------------------------------------------------------- */
/*                              RDM SET callbacks                               */
/* -------------------------------------------------------------------------- */

#ifdef RDM_SUPPORT_REQUIRED
/** @brief SET IDENTIFY_DEVICE callback (ESTA RDM). */
void RdmExtSetIdentify(rdm_answer_t *rdmImp, bool set);
#endif

#ifdef RDM_SUPPORT_BASIC
/** @brief SET FACTORY_DEFAULTS callback (ESTA RDM). */
void RdmExtSetFactoryDefaults(rdm_answer_t *rdmImp);

/** @brief SET DEVICE_LABEL callback (ESTA RDM). */
void RdmExtSetDeviceLabel(rdm_answer_t *rdmImp, char *label, uint8_t labelLength);

/** @brief SET DMX_PERSONALITY callback (ESTA RDM). */
void RdmExtSetDmxPersonality(rdm_answer_t *rdmImp, uint8_t personality);

/** @brief SET DMX_START_ADDRESS callback (ESTA RDM). */
void RdmExtSetDmxStartaddress(rdm_answer_t *rdmImp, uint16_t dmxStartAddress);

/** @brief SET RESET_DEVICE callback (ESTA RDM). */
void RdmExtSetResetDevice(rdm_answer_t *rdmImp, rdm_reset_t reset);

/** @brief SET POWER_STATE callback (ESTA RDM). */
void RdmExtSetPowerState(rdm_answer_t *rdmImp, rdm_powerstate_t powerState);

/** @brief SET custom PID callback (manufacturer specific). */
void RdmExtSetCustomPid(rdm_answer_t *rdmImp, uint16_t pid, uint8_t *data, uint16_t pdl);
#endif

#ifdef RDM_SUPPORT_CUSTOM_IQ
/** @brief SET IQ input source callback (manufacturer specific). */
void RdmExtSetIqInputSource(rdm_answer_t *rdmImp, rdm_iq_inputsource_t inputSource);

/** @brief SET IQ input state callback (manufacturer specific). */
void RdmExtSetIqInputState(rdm_answer_t *rdmImp, rdm_iq_set_input_state_t inputState);

/** @brief SET IQ popup callback (manufacturer specific). */
void RdmExtSetIqPopup(rdm_answer_t *rdmImp, rdm_iq_popup_t popup);

/** @brief SET IQ emergency mode callback (manufacturer specific). */
void RdmExtSetIqEmergencyMode(rdm_answer_t *rdmImp, bool emergencyMode);

/** @brief SET IQ battery runtime callback (manufacturer specific). */
void RdmExtSetIqBatteryRuntime(rdm_answer_t *rdmImp, rdm_iq_battery_runtime_t runTime);

/** @brief SET IQ anti-theft callback (manufacturer specific). */
void RdmExtSetIqAntitheftMode(rdm_answer_t *rdmImp, rdm_iq_antitheft_t antitheft);
#endif

#ifdef RDM_SUPPORT_SELFTEST
/** @brief SET PERFORM_SELFTEST callback (ESTA RDM). */
void RdmExtSetPerformSelftest(rdm_answer_t *rdmImp, uint8_t selftest);
#endif

#ifdef RDM_SUPPORT_CUSTOM_LEDCAL
/** @brief SET custom LED calibration send callback (manufacturer specific). */
void RdmExtSetCustomAdjustLedCalibrationSend(rdm_answer_t *rdmImp, rdm_adjustledcalibration_send_set_t set,
											 uint8_t *data, uint8_t chunkLength,
											 custom_adjust_cal_send_response_t *response);

/** @brief SET custom LED calibration data callback (manufacturer specific). */
void RdmExtSetCustomAdjustLedCalibrationData(rdm_answer_t *rdmImp, rdm_adjustledcalibration_set_t responseBuf);
#endif

#ifdef RDM_SUPPORT_CUSTOM_SERVICEHOURS
/** @brief SET custom service hours callback (manufacturer specific). */
void RdmExtSetCustomServiceHours(rdm_answer_t *rdmImp, uint32_t counter);
#endif

#ifdef RDM_SUPPORT_EXTENDED
/** @brief SET LANGUAGE callback (ESTA RDM). */
void RdmExtSetLanguage(rdm_answer_t *rdmImp, rdm_language_set_t language);

/** @brief SET DISPLAY_INVERT callback (ESTA RDM). */
void RdmExtSetDisplayInvert(rdm_answer_t *rdmImp, rdm_display_invert_t displayInvert);

/** @brief SET DISPLAY_LEVEL callback (ESTA RDM). */
void RdmExtSetDisplayLevel(rdm_answer_t *rdmImp, uint8_t displayLevel);

/** @brief SET PAN_INVERT callback (ESTA RDM). */
void RdmExtSetPanInvert(rdm_answer_t *rdmImp, bool invert);

/** @brief SET TILT_INVERT callback (ESTA RDM). */
void RdmExtSetTiltInvert(rdm_answer_t *rdmImp, bool invert);

/** @brief SET CAPTURE_PRESET callback (ESTA RDM). */
void RdmExtSetCapturePreset(rdm_answer_t *rdmImp, rdm_capture_preset_set_t preset);

/** @brief SET REAL_TIME_CLOCK callback (ESTA RDM). */
void RdmExtSetRealTimeClock(rdm_answer_t *rdmImp, uint64_t rtc);

/** @brief SET PRESET_PLAYBACK callback (ESTA RDM). */
void RdmExtSetPresetPlayback(rdm_answer_t *rdmImp, rdm_preset_playback_response_t preset);
#endif

#ifdef RDM_SUPPORT_DIMMER
/** @brief SET MINIMUM_LEVEL callback (ESTA RDM). */
void RdmExtSetMinimumLevel(rdm_answer_t *rdmImp, rdm_minimum_response_t minimumLevel);

/** @brief SET MAXIMUM_LEVEL callback (ESTA RDM). */
void RdmExtSetMaximumLevel(rdm_answer_t *rdmImp, uint16_t maximumLevel);

/** @brief SET CURVE callback (ESTA RDM). */
void RdmExtSetCurve(rdm_answer_t *rdmImp, uint8_t curve);

/** @brief SET OUTPUT_RESPONSE_TIME callback (ESTA RDM). */
void RdmExtSetOutputResponseTime(rdm_answer_t *rdmImp, uint8_t responseTime);

/** @brief SET MODULATION_FREQUENCY callback (ESTA RDM). */
void RdmExtSetModulationFrequency(rdm_answer_t *rdmImp, uint8_t modulationFrequency);
#endif

#ifdef RDM_SUPPORT_LAMP
/** @brief SET LAMP_ON_MODE callback (ESTA RDM). */
void RdmExtSetLampOnMode(rdm_answer_t *rdmImp, rdm_lamp_on_mode_t lamp_mode);
#endif

#ifdef RDM_SUPPORT_COLORCONVERTER
/** @brief SET color converter license callback (manufacturer specific). */
void RdmExtSetColorConverterLicense(rdm_answer_t *rdmImp, rdm_colorconverter_license_t key);

/** @brief SET color converter calibration command callback (manufacturer specific). */
void RdmExtSetColorConverterCalibration(rdm_answer_t *rdmImp, rdm_colorconverter_cal_command_t command,
										uint16_t dataOffset, uint16_t *calData, uint8_t length);

/** @brief SET color converter control callback (manufacturer specific). */
void RdmExtSetColorConverterControl(rdm_answer_t *rdmImp, uint8_t ledIndex, uint8_t dieIndex, uint16_t dimmerValue);
#endif

#ifdef RDM_SUPPORT_CUSTOM_DFU
/** @brief SET custom DFU start callback (manufacturer specific). */
void RdmExtSetCustomDfuStart(rdm_answer_t *rdmImp, uint8_t dfuPacket8XLength,const uint8_t *startData, rdm_custom_response_dfu_start_t *response);

/** @brief SET custom DFU send callback (manufacturer specific). */
void RdmExtSetCustomDfuSend(rdm_answer_t *rdmImp, uint16_t packetIndex, uint8_t *data, uint8_t length);

/** @brief SET custom DFU end callback (manufacturer specific). */
void RdmExtSetCustomDfuEnd(rdm_answer_t *rdmImp, rdm_custom_response_dfu_status_code_t *response);
#endif

/** @} */ // end of group rdmhandler_api

/* -------------------------------------------------------------------------- */
/*                             Integration quick guide                          */
/* -------------------------------------------------------------------------- */

/**
 * @page rdmhandler_integration RDMHandler integration (quick guide)
 *
 * @section rdmhandler_integration_steps Steps
 * 1. Define your feature macros (`RDM_SUPPORT_*`) to enable only needed PIDs.
 * 2. Create one @ref rdm_library_instance_t instance (static or global).
 * 3. Call @ref rdm_init and provide your `rdm_answer_callback` to transmit responses.
 * 4. Implement required callbacks `RdmExtGet*` / `RdmExtSet*` for your enabled features.
 * 5. In your UART/DMX-RDM receive code, pass incoming RDM packets into @ref rdm_decodeMessage.
 *
 * @section rdmhandler_integration_supported_parameters Supported Parameters
 * Implement @ref RdmExtGetSupportedParameter and add all supported PIDs manually.
 * This list must match your implemented callbacks and enabled features.
 *
 * @section rdmhandler_integration_common_rules Common rules
 * - Not thread-safe: Do not call @ref rdm_decodeMessage concurrently.
 * - Do not store `responseBuf` pointers. They are valid only during the callback.
 * - Default answer is success. Set @ref rdm_answer_t only in error cases.
 */

#endif