/**
 ******************************************************************************
 * @file           : button.c
 * @brief          : Simple example to implement a button with short / long press
 ******************************************************************************
 * @attention
 *
 * Copyright (c) Kuenzler Technologies GbR
 * All rights reserved.
 *
 *
 ******************************************************************************
 */

/*==============================================================================*/
/* Includes                                                                  	*/
/*==============================================================================*/
#include "button.h"

#include "stdlib.h"
#include "string.h"
#ifdef RTTLOGGINGSET
#include "loggerservice.h"
#endif
/*==============================================================================*/
/* Defines & Macros                                                           	*/
/*==============================================================================*/
#ifdef RTTLOGGINGSET
// LOGGER ENABLE ------------
#define LOGENABLE 0
#endif

#define DEBOUNCE_DELAY_MS    20
#define SINGLECLICK_DELAY_MS 250
#define OTHERUP_DELAY_MS     300

/*==============================================================================*/
/* Private variables                                                          	*/
/*==============================================================================*/

/*==============================================================================*/
/* Private functions                                                          	*/
/*==============================================================================*/

void InitButton(button_instance_t *instance, uint8_t instanceId, button_callbacks_t callbacks,
				uint16_t _longPressTimeMs, bool _activeStatePressed, bool useTask)
{
	memset(instance, 0, sizeof(button_instance_t));
	instance->btnState           = StateIdle;
	instance->longPressTimeMs    = _longPressTimeMs;
	instance->activeStatePressed = _activeStatePressed;
	instance->handler            = callbacks;
	instance->id                 = instanceId;
	instance->useTask            = useTask;
#ifdef RTTLOGGINGSET
	LOGGER_INFO("Init");
#endif
}

void ButtonTask(button_instance_t *instance)
{
	if(instance->flags.eventButtonShortPress)
	{
		instance->flags.eventButtonShortPress = false;
		if(instance->handler.ButtonShortPress != NULL)
		{
			instance->handler.ButtonShortPress();
		}
	}
	if(instance->flags.eventButtonLongPress)
	{
		instance->flags.eventButtonLongPress = false;
		if(instance->handler.ButtonLongPress != NULL)
		{
			instance->handler.ButtonLongPress(instance->longPressCounter);
		}
	}
	if(instance->flags.eventButtonDoubleClick)
	{
		instance->flags.eventButtonDoubleClick = false;
		if(instance->handler.ButtonDoubleClick != NULL)
		{
			instance->handler.ButtonDoubleClick();
		}
	}
	if(instance->flags.eventButtonPressed)
	{
		instance->flags.eventButtonPressed = false;
		if(instance->handler.ButtonPressed != NULL)
		{
			instance->handler.ButtonPressed();
		}
	}
	if(instance->flags.eventButtonPressReleased)
	{
		instance->flags.eventButtonPressReleased = false;
		if(instance->handler.ButtonReleased != NULL)
		{
			instance->handler.ButtonReleased(instance->longPressCounter);
		}
	}
}

void ButtonInteruptIrq(button_instance_t *instance, bool pinSet)
{
	switch(instance->btnState)
	{
	case StateIdle:
		if(pinSet == instance->activeStatePressed)
		{
			// Start button routine
			instance->btnState         = StateDebounce;
			instance->longPressCounter = 0;
#ifdef RTTLOGGINGSET
			LOGGER_INFO("StateDebounce");
#endif
			HardwareButtonTimerStopStartWaitTime(DEBOUNCE_DELAY_MS, instance->id);
		}
		break;
	case StateDebounce:
		// Do nothing and wait for debounce
		break;
	case StatePressed:
		if(pinSet != instance->activeStatePressed)
		{
			// Not pressed anymore and no long press detected. Debounce off
			instance->btnState = StateClickUp;
#ifdef RTTLOGGINGSET
			LOGGER_INFO("StateClickUp");
#endif
			HardwareButtonTimerStopStartWaitTime(DEBOUNCE_DELAY_MS, instance->id);
		}
		break;
	case StateClickUp:
		// Do nothing and wait for debounce
		break;
	case StateClickIdle:
		if(pinSet == instance->activeStatePressed)
		{
			// Button pressed again before SingleClick detected
			instance->btnState = StateDoubleClickDebounce;
#ifdef RTTLOGGINGSET
			LOGGER_INFO("StateDoubleClickDebounce");
#endif
			HardwareButtonTimerStopStartWaitTime(DEBOUNCE_DELAY_MS, instance->id);
		}
		break;
	case StateDoubleClickDebounce:
		// Do nothing and wait for debounce
		break;
	case StateDoubleClick:
		if(pinSet != instance->activeStatePressed)
		{
			// Button not pressed anymore
			instance->btnState = StateOtherUp;
#ifdef RTTLOGGINGSET
			LOGGER_INFO("StateOtherUp");
#endif
			HardwareButtonTimerStopStartWaitTime(DEBOUNCE_DELAY_MS, instance->id);
		}
		break;
	case StateLongClickCount:
		if(pinSet != instance->activeStatePressed)
		{
			// Button not pressed anymore
			instance->btnState = StateOtherUp;
#ifdef RTTLOGGINGSET
			LOGGER_INFO("StateOtherUp");
#endif
			HardwareButtonTimerStopStartWaitTime(DEBOUNCE_DELAY_MS, instance->id);
		}
		break;
	case StateOtherUp:
		// Do nothing and wait for debounce
		break;
	}
}
void ButtonWaitTimeElapsed(button_instance_t *instance)
{
	switch(instance->btnState)
	{
	case StateIdle:
		// Should not occur
		break;
	case StateDebounce:
		// Check if button is still pressed
		if(HardwareButtonGpioReadPin(instance->id) == instance->activeStatePressed)
		{
			// Signal Button pressed
			instance->flags.eventButtonPressed = true;
			// Set State to pressed
			instance->btnState = StatePressed;
#ifdef RTTLOGGINGSET
			LOGGER_INFO("StatePressed");
#endif
			// Measure Time in order to detect longpress 1
			HardwareButtonTimerStopStartWaitTime(instance->longPressTimeMs, instance->id);
		}
		else
		{
			instance->btnState = StateIdle;
		}
		break;
	case StatePressed:
		// Check if button is still pressed
		if(HardwareButtonGpioReadPin(instance->id) == instance->activeStatePressed)
		{
			// Button still pressed --> Long press detected
			instance->btnState = StateLongClickCount;
#ifdef RTTLOGGINGSET
			LOGGER_INFO("StateLongClickCount");
#endif
			instance->longPressCounter           = 1;
			instance->flags.eventButtonLongPress = true;
			// Measure Time in order to detect longpress 2
			HardwareButtonTimerStopStartWaitTime(instance->longPressTimeMs, instance->id);
		}
		else
		{
			// Button is not pressed anymore. Should not occur because ButtonInteruptIrq should have detected
			instance->btnState = StateIdle;
		}
		break;
	case StateLongClickCount:
		// Check if button is still pressed
		if(HardwareButtonGpioReadPin(instance->id) == instance->activeStatePressed)
		{
			// Button still pressed --> Long press detected
			instance->longPressCounter++;
#ifdef RTTLOGGINGSET
			LOGGER_INFO("Counting Longpress %d", instance->longPressCounter);
#endif
			instance->flags.eventButtonLongPress = true;
			HardwareButtonTimerStopStartWaitTime(instance->longPressTimeMs, instance->id);
		}
		else
		{
			// Button is not pressed anymore. Should not occur because ButtonInteruptIrq should have detected
			instance->btnState = StateIdle;
		}
		break;
	case StateClickUp:
		instance->btnState = StateClickIdle;
#ifdef RTTLOGGINGSET
		LOGGER_INFO("StateClickIdle");
#endif
		instance->flags.eventButtonPressReleased = true;
		// Measure Time in order to detect Single or double click
		HardwareButtonTimerStopStartWaitTime(SINGLECLICK_DELAY_MS, instance->id);
		break;
	case StateClickIdle:
		// It was a single click
		instance->flags.eventButtonShortPress = true;
		instance->btnState                    = StateIdle;
#ifdef RTTLOGGINGSET
		LOGGER_INFO("StateIdle");
#endif
		break;
	case StateDoubleClickDebounce:
		if(HardwareButtonGpioReadPin(instance->id) == instance->activeStatePressed)
		{
			// Button still pressed --> Double press detected
			instance->btnState = StateDoubleClick;
#ifdef RTTLOGGINGSET
			LOGGER_INFO("StateDoubleClick");
#endif
			instance->flags.eventButtonDoubleClick = true;
		}
		else
		{
			instance->btnState = StateIdle;
#ifdef RTTLOGGINGSET
			LOGGER_INFO("StateIdle");
#endif
			instance->flags.eventButtonPressReleased = true;
		}
		break;
	case StateDoubleClick:
		// Should not occur
		instance->btnState = StateIdle;
#ifdef RTTLOGGINGSET
		LOGGER_INFO("StateIdle");
#endif
		break;
	case StateOtherUp:
		// Go back to idle state
		instance->flags.eventButtonPressReleased = true;
		instance->btnState                       = StateIdle;
#ifdef RTTLOGGINGSET
		LOGGER_INFO("StateIdle");
#endif
		break;
	}
	if(!(instance->useTask))
	{
		ButtonTask(instance);
	}
}
