/**
  ******************************************************************************
  * @file           : output.c
  * @brief          : Simple example for setting output
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Kuenzler Technologies GbR
  * All rights reserved.
  *
  *
  ******************************************************************************
*/

/*==============================================================================*/
/* Includes                                                                  	*/
/*==============================================================================*/
#include "main.h"
#include "output.h"
#include "output_setting.h"
#include "string.h"
#include "stdio.h"
#include "defines.h"
#include "variables.h"
#include "stdlib.h"

/*==============================================================================*/
/* Defines & Macros                                                           	*/
/*==============================================================================*/
#define OUTPUT_REFRESH_RATE_US 910
#define FADETRESHOLD (0xFFFF/5)

/*==============================================================================*/
/* Private functions                                                          	*/
/*==============================================================================*/

/*==============================================================================*/
/* Private variables                                                          	*/
/*==============================================================================*/
static bool enabled;

#define LED_RESET_LENGTH 50
#define LED_DATA_LENGTH (8 * OUTPUT_LED_COLORS *OUTPUT_LED_COUNT) + LED_RESET_LENGTH

volatile static uint16_t pwmData[LED_DATA_LENGTH];
static bool pixelSending = false;

static bool turnOnAnimation = false;
static bool turnOffAnimation = false;
static uint8_t ledCounter = 0;
static uint32_t lastTick = 0;

#define TIMEMSSANIMATION 80

/*==============================================================================*/
/* Function definitions                                                     		*/
/*==============================================================================*/

void InitOutput()
{
	InitOutputSetting();
}

bool OutputTask(uint32_t tickMs)
{
	OutputSettingTask(tickMs);
	if(turnOnAnimation)
	{
		if(lastTick + TIMEMSSANIMATION < tickMs)
		{
			if(ledCounter < OUTPUT_LED_COUNT)
			{
				lastTick = tickMs;
				ledCounter++;
				// Turn On leds
				uint8_t data[OUTPUT_LED_COUNT];
				memset(data,0,sizeof(data));
				for(uint8_t i=0; i< ledCounter;i++)
				{
					data[i] = 0xFF;
				}
				OutputSetRaw(data,data,data, true);
			}
			else
			{
				// Turn leds off and stop animation
				uint8_t data[OUTPUT_LED_COUNT];
				memset(data,0,sizeof(data));
				OutputSetRaw(data,data,data, true);
				turnOnAnimation = false;
				lastTick = tickMs;
			}
		}
		return true;
	}
	else if(turnOffAnimation)
	{
		if(lastTick + TIMEMSSANIMATION < tickMs)
		{
			if(ledCounter > 0)
			{
				lastTick = tickMs;
				ledCounter--;
				// Turn On leds
				uint8_t data[OUTPUT_LED_COUNT];
				memset(data,0,sizeof(data));
				for(uint8_t i=0; i< ledCounter;i++)
				{
					data[i] = 0xFF;
				}
				OutputSetRaw(data,data,data, true);
			}
			else
			{
				// Turn leds off and stop animation
				HardwareOutputTimPwmStop();
				pixelSending = false;
				turnOffAnimation = false;
				lastTick = tickMs;
				HardwareOutputGpioSetPowerEnablePin(false);
			}
		}
		return true;
	}
	return false;
}

void OutputEnable(bool _enable)
{
	enabled = _enable;
	if(enabled)
	{
		// Enable output
		HardwareOutputGpioSetPowerEnablePin(true);
		uint8_t data[OUTPUT_LED_COUNT];
		memset(data,0x0,sizeof(data));
		turnOnAnimation = true;
		ledCounter = 0;
		OutputSetRaw(data,data,data, true);
	}
	else
	{
		turnOffAnimation = true;
		ledCounter = OUTPUT_LED_COUNT;
		uint8_t data[OUTPUT_LED_COUNT];
		memset(data,0xFF,sizeof(data));
		turnOffAnimation = true;
	}
}

void OutputSetRaw(uint8_t* _red,uint8_t* _green,uint8_t* _blue, bool animation)
{
	uint32_t indx=0;
	uint32_t color;
	if(!animation && (turnOnAnimation || turnOffAnimation))
	{
		return;
	}

	OutputSettingGetColorOverride(_red, _green, _blue, enabled);

	for (int led= 0; led<OUTPUT_LED_COUNT; led++)
	{
		color = (( (uint32_t)(_red[led]) <<16) | (((uint32_t)_green[led])<<8) | (((uint32_t)_blue[led])));
		for (int i=((8 * OUTPUT_LED_COLORS) -1); i>=0; i--)
		{
			if (color&(1<<i))
			{
				pwmData[indx] = 40;  // 2/3 of 90
			}

			else pwmData[indx] = 20;  // 1/3 of 90

			indx++;
		}
	}

	for (int i=0; i<LED_RESET_LENGTH; i++)
	{
		pwmData[indx] = 0;
		indx++;
	}
	if(!pixelSending)
	{
		if(!HardwareOutputTimPwmStart((uint16_t*)pwmData, LED_DATA_LENGTH))
		{
			return;
		}
		pixelSending = true;
	}
}
