/**
  ******************************************************************************
  * @file           : dmx_implementation.c
  * @brief          : Simple example for showing dmx functionality
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Kuenzler Technologies GbR
  * All rights reserved.
  *
  *
  ******************************************************************************
*/

/*==============================================================================*/
/* Includes                                                                  	*/
/*==============================================================================*/
#include "dmx_implementation.h"
#include "variables.h"
#include "string.h"
#include "stdio.h"
#include "output.h"

/*==============================================================================*/
/* Defines & Macros                                                           	*/
/*==============================================================================*/

/*==============================================================================*/
/* Private functions                                                          	*/
/*==============================================================================*/
static void SetMode1(const uint8_t* dmx,uint16_t length);
static void SetMode2(const uint8_t* dmx,uint16_t length);
static void SetMode3(const uint8_t* dmx,uint16_t length);
static void SetMode255(const uint8_t* dmx,uint16_t length);
static void UpdateDmx(uint8_t personality, const  uint8_t* dmx,uint16_t length);


/*==============================================================================*/
/* Private variables                                                          	*/
/*==============================================================================*/

/*==============================================================================*/
/* Function definitions                                                     	*/
/*==============================================================================*/

void InitDmxImp()
{

}

void DmxImpTask(uint32_t tickMs)
{
	(void)tickMs;
}

void SetDmx(uint8_t personality,const  uint8_t* dmx,uint16_t length)
{
	/* iQ.Controller provides the personality within the frame in order to switch between function without changing dmx settings  - iQ.Controller 提供框架内的个性，以便在不更改 dmx 设置的情况下在功能之间切换*/
	/* if 0 use dmx settings personality as stated in the documentation - 如果 0 使用 dmx 设置个性，如文档中所述 */
	if(personality == 0)
	{
		UpdateDmx(get_dmxSettings()->dmxPersonality,dmx,length);
	}
	else
	{
		UpdateDmx(personality,dmx,length);
	}
}

static void UpdateDmx(uint8_t personality, const uint8_t* dmx,uint16_t length)
{
	switch(personality)
	{
		case 1: // RGB Basic Mode
			SetMode1(dmx, length);
		break;
		case 2: // RGB Advanced Mode
			SetMode2(dmx, length);
		break;
		case 3: // RGB Pixel Mode
			SetMode3(dmx, length);
		break;
		case 255: // App Control Mode
			SetMode255(dmx, length);
		break;
	}
}

/* Use DMX Mode 1 - 使用 DMX 模式 1 */
static void SetMode1(const uint8_t* dmx,uint16_t length)
{
	uint8_t red[OUTPUT_LED_COUNT];
	uint8_t green[OUTPUT_LED_COUNT];
	uint8_t blue[OUTPUT_LED_COUNT];
	
	// Mode RGB Control
	uint16_t address = get_dmxSettings()->dmxStartadress -1;
	if(length < (address + RDM_FOOTPRINTS[0]))
	{
		return;
	}

	for(uint8_t i = 0; i< OUTPUT_LED_COUNT; i++)
	{
		red[i] = dmx[address];
		green[i] = dmx[address + 1];
		blue[i] = dmx[address + 2];
	}

	/* Set LED output - 设置 LED 输出*/
	OutputSetRaw(red,green,blue, false);
}

static void SetMode2(const uint8_t* dmx,uint16_t length)
{
	uint8_t red[OUTPUT_LED_COUNT];
	uint8_t green[OUTPUT_LED_COUNT];
	uint8_t blue[OUTPUT_LED_COUNT];

	// Mode RGBW Control
	uint16_t address = get_dmxSettings()->dmxStartadress -1;
	if(length < (address + RDM_FOOTPRINTS[1]))
	{
		return;
	}
	
	uint8_t masterDimmer = dmx[address];

	for(uint8_t i = 0; i< OUTPUT_LED_COUNT; i++)
	{
		red[i] = (dmx[address + 1] * masterDimmer)/0xFF;
		green[i] = (dmx[address + 2] * masterDimmer)/0xFF;
		blue[i] = (dmx[address + 3] * masterDimmer)/0xFF;
	}

	/* Set LED output - 设置 LED 输出*/
	OutputSetRaw(red,green,blue, false);
}

static void SetMode3(const uint8_t* dmx,uint16_t length)
{
	uint8_t red[OUTPUT_LED_COUNT];
	uint8_t green[OUTPUT_LED_COUNT];
	uint8_t blue[OUTPUT_LED_COUNT];

	// Mode RGBW Control
	uint16_t address = get_dmxSettings()->dmxStartadress -1;
	if(length < (address + RDM_FOOTPRINTS[2]))
	{
		return;
	}

	for(uint8_t i = 0; i< OUTPUT_LED_COUNT; i++)
	{
		red[i] = dmx[address + i * 3];
		green[i] = dmx[address + i * 3 + 1];
		blue[i] = dmx[address + i * 3 + 2];
	}

	/* Set LED output - 设置 LED 输出*/
	OutputSetRaw(red,green,blue, false);
}


/* Use DMX Mode 255 - 使用 DMX 模式 255 */
static void SetMode255(const uint8_t* dmx,uint16_t length)
{
	/* Magic Mode for App Control - 应用程序控制的神奇模式 */
	/* There are some mandatory channels and behaviors -  有一些强制性的渠道和行为*/
	/* These are explained in detail in the documentation - 这些在文档中有详细解释*/
	/* Layout is not important due to dmx like configurations for each fixture in the App itself - 布局并不重要，因为应用程序本身的每个灯具都有类似 dmx 的配置*/
	/* Do not apply dmxaddress and use default configuration for the fixture while using this mode. - 使用此模式时，请勿应用 dmxaddress 并使用灯具的默认配置。*/

	/* Master Dimmer is not needed - 不需要主调光器 */
	/* Shutter is not needed. - 不需要快门 */
	/* The App will virtually create shutter using the dimmer for each pixel. To ensure that the devices all behave the same way - 该应用程序将使用每个像素的调光器虚拟地创建快门。确保所有设备的行为方式相同*/
	/* Therefore it is important that there is no fade time longer than 20ms for the led - 因此，LED 的衰落时间不得超过 20ms，这一点很重要。*/
	/* Dimmer of the colors should be linear and 16 bit if possible. (Coarse and fine) - 如果可能的话，颜色调光器应该是线性的并且是 16 位的。 （粗、细）*/
	/* If the fixture has single pixel control they should be provided like [Red0,Green0,Blue0,White0,Red1,Green1,Blue1,White1,..] - 如果灯具具有单像素控制，则应提供类似 [Red0,Green0,Blue0,White0,Red1,Green1,Blue1,White1,..]*/
	uint8_t red[OUTPUT_LED_COUNT];
	uint8_t green[OUTPUT_LED_COUNT];
	uint8_t blue[OUTPUT_LED_COUNT];
	
	if(length < 8)
	{
		return;
	}
	
	for(uint8_t i = 0; i< OUTPUT_LED_COUNT; i++)
	{
		red[i] = dmx[i * 3];
		green[i] = dmx[i * 3 + 1];
		blue[i] = dmx[i * 3 + 2];
	}

	/* Set LED output - 设置 LED 输出*/
	OutputSetRaw(red,green,blue,false);

}
