/* Header */
/**
 ******************************************************************************
 * @file           : spitypeiqmesh.c
 * @brief          : Spi Handler iQMesh
 ******************************************************************************
 */
/* Header */
/* Includes ------------------------------------------------------------------*/
#include "spi_iqmesh.h"

#include <stdio.h>
#include <stdlib.h>

/* Private includes ----------------------------------------------------------*/

/* Private typedef -----------------------------------------------------------*/
typedef enum {
	SPITXSTATE_IDLE,
	SPITXSTATE_WAIT,
	SPITXSTATE_STARTCOMMANDCS,
	SPITXSTATE_STARTCOMMANDDATA,
	SPITXSTATE_COMMANDSENT,
	SPITXSTATE_IRQCS,
	SPITXSTATE_IRQDATA,
	SPITXSTATE_DATASENT,
	SPITXSTATE_ERR,
} spi_tx_state_t;

typedef struct
{
	uint32_t timestamp;
	uint8_t *txBuf;
	spi_message_header_t msgtx;
	spi_message_header_t msgrx;
	spi_tx_state_t state;
	bool timeoutSet;
	bool valid;
} spi_state;

/* Private define ------------------------------------------------------------*/
#define TIMEOUTMS  10
#define WAITTIMEMS 1
/* Private macro -------------------------------------------------------------*/

/* Private variables ---------------------------------------------------------*/
static spi_state state;
static spi_iqmesh_callbacks_t handler;

/* Private function prototypes -----------------------------------------------*/
static void End(void);
static void CompletionEvent(uint8_t *buffer, uint16_t length);
static bool AllocateTxBuffer(uint16_t length);
static void FreeTxBuffer(void);
/* Private user code ---------------------------------------------------------*/

void spiIqMesh_init(spi_iqmesh_callbacks_t callbacks)
{
	handler = callbacks;
	memset(&state, 0, sizeof(state));
	FreeTxBuffer();
	if(handler.SPISetCs != NULL)
	{
		handler.SPISetCs(true);
	}
}

bool spiIqMesh_PrepareData(spi_message_header_t header, uint8_t **bufPointer)
{
	if(state.state != SPITXSTATE_IDLE)
	{
		// communication ongoing
		return false;
	}
	if(!AllocateTxBuffer(header.length))
	{
		return false;
	}
	state.msgtx = header;
	*bufPointer = state.txBuf;
	return true;
}

bool spiIqMesh_SetData(bool irqSet)
{
	// If IRQ was set controller is ready to talk
	if(irqSet)
	{
		// Talk directly
		state.state = SPITXSTATE_STARTCOMMANDCS;
		// Start sending
		if(handler.SPISetCs != NULL)
		{
			state.timeoutSet = false;
			handler.SPISetCs(false);
		}
		else
		{
			return false;
		}
	}
	else
	{
		// Wait first to make sure he is ready
		state.timeoutSet = false;
		state.state      = SPITXSTATE_WAIT;
	}
	return true;
}

void spiIqMesh_extIrqInterupt(iqmesh_irq_edge_t edge)
{
	switch(state.state)
	{
	case SPITXSTATE_IDLE:
		if(edge == SPIEXT_IRQ_EDGE_FALLING)
		{
			if(handler.IrQFlag != NULL)
			{
				handler.IrQFlag();
			}
		}
		break;
	case SPITXSTATE_STARTCOMMANDCS:
		// Should not occur
		break;
	case SPITXSTATE_STARTCOMMANDDATA:
		// Should not occur
		break;
	case SPITXSTATE_COMMANDSENT:
		if(edge == SPIEXT_IRQ_EDGE_FALLING)
		{
			state.state = SPITXSTATE_IRQCS;
			if(handler.SPISetCs != NULL)
			{
				handler.SPISetCs(false);
			}
		}
		break;
	case SPITXSTATE_IRQCS:
		// Should not occur
		break;
	case SPITXSTATE_IRQDATA:
		// Should not occur
		break;
	case SPITXSTATE_DATASENT:
		if(edge == SPIEXT_IRQ_EDGE_RISING)
		{
			End();
		}
		if(edge == SPIEXT_IRQ_EDGE_FALLING)
		{
			// Should not occur
			state.valid = false;
			End();
		}
		break;
	case SPITXSTATE_ERR:
		break;
	case SPITXSTATE_WAIT:
		break;
	}
}

void spiIqMesh_extSpiEvent(spi_event_t event, uint8_t *buffer, uint16_t length)
{
	if(handler.SPISetCs != NULL)
	{
		handler.SPISetCs(true);
	}
	switch(event)
	{
	case SPI_EVENT_COMPLETION:
		CompletionEvent(buffer, length);
		break;
	case SPI_EVENT_ERROR:
		state.valid = false;
		End();
		break;
	}
}

bool spiIqMesh_Task(uint32_t tickMs)
{
	if(state.state != SPITXSTATE_IDLE)
	{
		if(state.state == SPITXSTATE_STARTCOMMANDCS)
		{
			if(handler.SPISetTxRx != NULL)
			{
				handler.SPISetTxRx((uint8_t *)&state.msgtx.cmd, 1);
			}
			state.state = SPITXSTATE_STARTCOMMANDDATA;
		}
		else if(state.state == SPITXSTATE_IRQCS)
		{
			if(handler.SPISetTxRx != NULL)
			{
				handler.SPISetTxRx(state.txBuf, state.msgtx.length);
			}
			state.state = SPITXSTATE_IRQDATA;
		}
		if(!state.timeoutSet)
		{
			state.timeoutSet = true;
			state.timestamp  = tickMs;
		}
		else if(state.state == SPITXSTATE_WAIT)
		{
			if(state.timestamp > tickMs || tickMs >= state.timestamp + WAITTIMEMS)
			{
				// Start sending command
				state.timeoutSet = false;
				state.state      = SPITXSTATE_STARTCOMMANDCS;
				// Start sending
				if(handler.SPISetCs != NULL)
				{
					state.timeoutSet = false;
					handler.SPISetCs(false);
				}
				else
				{
					return false;
				}
			}
		}
		else if(state.timestamp > tickMs || tickMs >= state.timestamp + TIMEOUTMS)
		{
			End();
		}
		return false;
	}
	else
	{
		return true;
	}
}

bool spiIqMesh_IsIdle()
{
	return state.state == SPITXSTATE_IDLE;
}

static void CompletionEvent(uint8_t *buffer, uint16_t length)
{
	switch(state.state)
	{
	case SPITXSTATE_IDLE:
		// Should not occur
		break;
	case SPITXSTATE_STARTCOMMANDCS:
		// Should not occur
		break;
	case SPITXSTATE_STARTCOMMANDDATA:
		state.state = SPITXSTATE_COMMANDSENT;
		if(length > 0)
		{
			if(buffer[0] != 0xFF)
			{
				if(handler.IrQ != NULL)
				{
					handler.IrQ(buffer[0]);
				}
			}
			else
			{
				state.state = SPITXSTATE_ERR;
			}
		}
		break;
	case SPITXSTATE_COMMANDSENT:
		// Should not occur
		break;
	case SPITXSTATE_IRQCS:
		// Should not occur
		break;
	case SPITXSTATE_IRQDATA:
		state.state        = SPITXSTATE_DATASENT;
		state.msgrx.cmd    = state.msgtx.cmd;
		state.msgrx.length = length;
		FreeTxBuffer();
		// IRQ 0xFF is not valid - all flags occured at the same time is not possible. Throw frame away
		if(buffer[0] != 0xFF)
		{
			if(handler.DataRx != NULL)
			{
				handler.DataRx(state.msgrx, buffer);
			}
			state.valid = true;
		}
		else
		{
			state.valid = false;
		}
		break;
	case SPITXSTATE_DATASENT:
		// Should not occur
		break;
	case SPITXSTATE_ERR:
		break;
	case SPITXSTATE_WAIT:
		break;
	}
}

static void End()
{
	state.timeoutSet = false;
	state.state      = SPITXSTATE_IDLE;
	FreeTxBuffer();
	if(handler.Finish != NULL)
	{
		handler.Finish(state.valid);
	}
}

static bool AllocateTxBuffer(uint16_t length)
{
	if(state.txBuf)
	{
		free(state.txBuf);
	}
	if(!state.txBuf)
	{
		state.txBuf = (uint8_t *)malloc(length);
		if(!state.txBuf)
		{
			return false;
		}
	}
	else
	{
		return false;
	}
	return true;
}

static void FreeTxBuffer()
{
	if(state.txBuf)
	{
		free(state.txBuf);
		state.txBuf = NULL;
	}
}
