/* Header */
/**
 ******************************************************************************
 * @file           : libIqMeshSpiInterface.c
 * @brief          : iQMesh Handler
 ******************************************************************************
 */
/* Header */
/* Includes ------------------------------------------------------------------*/
#include "libIqMeshSpiInterface.h"

#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>

#include "spi_iqmesh.h"
#include "string.h"
/* Private includes ----------------------------------------------------------*/

/* Private typedef -----------------------------------------------------------*/
typedef enum {
	IQMESH_SYSTEM_UNCONFIGURED           = 0,
	IQMESH_SYSTEM_GOTOSLEEP              = 1,
	IQMESH_SYSTEM_NORMAL                 = 2,
	IQMESH_SYSTEM_OPEN                   = 3,
	IQMESH_SYSTEM_OPEN_NON_ACTIVE_SOURCE = 4,
} interface_systemstate_t;

typedef enum {
	IQMESHSTATE_NOTCONFIGURED,
	IQMESHSTATE_RDMIDSET,
	IQMESHSTATE_RDMID_CHECK,
	IQMESHSTATE_VERSIONRECEIVED,
	IQMESHSTATE_CPUIDRECEIVED,
	IQMESHSTATE_NETWORKINFORECEIVED,
	IQMESHSTATE_CHECKIRQ,
	IQMESHSTATE_RUN,
	IQMESHSTATE_RUNSET,
	IQMESHSTATE_UPDATEIRQ,
} iq_mesh_state_t;

typedef struct
{
	uint32_t module_hardware_version;
	uint32_t module_firmware_version;
	uint32_t module_bootloader_version;
	uint16_t status_flags;
	iqMesh_information_t info;
	uint8_t module_rdm_id[6];
	uint8_t irq;
	uint8_t cpuId[8];
	iq_mesh_state_t state;
	iqmesh_visibility_t visibility;
	bool visibilitySet;
} iq_mesh_config_t;

typedef struct
{
	uint8_t *buffer;
	uint16_t length;
	bool set;
	uint8_t padding;
} dynamic_buf_t;

typedef struct
{
	dynamic_buf_t rxDfu;
	uint8_t dfuResponse;
	bool support;
	bool started;
	bool finished;
	bool responseSet;
	uint8_t padding[3];
} iq_dfu_state_t;

/* Private define ------------------------------------------------------------*/
#define DFUPAGELENGTH 128
#define DMXLENGTH     512

#define STATUS_IQ_LINKED    0x0001
#define STATUS_IQ_RF_LINKED 0x0002
#define STATUS_IQSTATE      0xFF00

#define CMD_READ_REG  0x00
#define CMD_WRITE_REG 0x40
#define CMD_READ_DMX  0x81
#define CMD_READ_RDM  0x83
#define CMD_READ_DFU  0x87
#define CMD_WRITE_DMX 0x91
#define CMD_WRITE_RDM 0x93
#define CMD_WRITE_DFU 0x97
#define CMD_NOP       0xFF

#define REG_SYSTEM_STATE        0x02
#define REG_START_DFU           0x05
#define REG_STATUS              0x06
#define REG_IRQ_FLAGS           0x08
#define REG_RDM_ID              0x0A
#define REG_NETWORK_INFORMATION 0x0D
#define REG_VERSION             0x0E
#define REG_IQ_CPUID            0x11

#define IRQ_SYS_RESTARTED 0x01
#define IRQ_RX_DMX        0x02
#define IRQ_RX_RDM        0x04
#define IRQ_NFC_WAKEUP    0x08
#define IRQ_RF_LINK       0x10
#define IRQ_DFU_START     0x20
#define IRQ_DFU_PACKET    0x40
#define IRQ_DFU_STOP      0x80
#define IRQ_DFU_FLAGS     (IRQ_DFU_START | IRQ_DFU_PACKET | IRQ_DFU_STOP)

#define RDM_FRAME_LENGTH 258

#define REG_STATUS_LENGTH              2
#define REG_IRQ_FLAGS_LENGTH           1
#define REG_RDM_ID_LENGTH              6
#define REG_NETWORK_INFORMATION_LENGTH 22
#define REG_VERSION_LENGTH             12
#define REG_IQ_CPUID_LENGTH            8

/* Private macro -------------------------------------------------------------*/

/* Private variables ---------------------------------------------------------*/
static uint8_t rdm_id[6];
static iq_mesh_config_t config;
static dynamic_buf_t rxDmx;
static dynamic_buf_t txDmx;
static dynamic_buf_t rxRdm;
static dynamic_buf_t txRdm;
static iq_dfu_state_t dfuState;
static bool unlink     = false;
static bool startDfu   = false;
static bool statusFlag = false;
static uint16_t iqSpecialDmxLength;
static uint8_t lastIqDmxMode;

/* Private function prototypes -----------------------------------------------*/
static void spi_irq_callback_t(uint8_t irq);
static void spi_irq_flag_callback_t(void);
static void spi_data_rx_callback_t(spi_message_header_t msg, uint8_t *buffer);
static void spi_finish_callback_t(bool success);

static void spi_settxrx_callback_t(uint8_t *buf, uint16_t length);
static void spi_setcs_callback_t(bool set);

static void Run(bool irq);
static bool HandleTxRdm(void);
static bool HandleTxDmx(void);
static bool HandleTxDfu(void);
static bool Handle_SysRestart(void);
static bool HandleRxRdm(void);
static bool HandleRxDmx(void);
static bool HandleDfuFlag(void);
static bool HandleDfuData(void);
static bool HandleNfcFlag(void);
static bool HandleRfLink(void);
static bool HandleStatus(void);
static bool CheckRxRdm(void);
static bool CheckRxDmx(void);
static bool CheckRxDfu(void);
static bool HandleUnlink(void);
static bool HandleStartDfu(void);
static void FreeBuffer(dynamic_buf_t *buf);
static bool AllocateBuffer(dynamic_buf_t *buf, uint16_t length);
static bool HandleVisibility();

// Handler
static void (*DmxReceivedHandler)(uint8_t personality, const uint8_t *buf, uint16_t length);
static void (*RdmReceivedHandler)(const uint8_t *buf, uint16_t length);
static void (*DfuFlagReceivedHandler)(iqmesh_dfu_flag_t);
static void (*DfuDataReceivedHandler)(uint16_t packetNr, const uint8_t *buf, uint16_t length);

/* Private user code ---------------------------------------------------------*/

bool iqMesh_init(const uint8_t *_rdm_id, uint16_t specialDmxLength, bool dfuSupport, iqmesh_callbacks_t callbacks)
{
	spi_iqmesh_callbacks_t spiCallbacks;
	// Init iqmesh
	FreeBuffer(&txDmx);
	FreeBuffer(&rxRdm);
	FreeBuffer(&txRdm);
	FreeBuffer(&(dfuState.rxDfu));
	dfuState.support   = dfuSupport;
	iqSpecialDmxLength = specialDmxLength + 1;  // include personality at start
	lastIqDmxMode      = 0;

	spiCallbacks.SPISetTxRx = &spi_settxrx_callback_t;
	spiCallbacks.SPISetCs   = &spi_setcs_callback_t;
	spiCallbacks.DataRx     = &spi_data_rx_callback_t;
	spiCallbacks.IrQ        = &spi_irq_callback_t;
	spiCallbacks.IrQFlag    = &spi_irq_flag_callback_t;
	spiCallbacks.Finish     = &spi_finish_callback_t;
	spiIqMesh_init(spiCallbacks);

	DmxReceivedHandler     = callbacks.DmxReceived;
	RdmReceivedHandler     = callbacks.RdmReceived;
	DfuFlagReceivedHandler = callbacks.DfuFlagReceived;
	DfuDataReceivedHandler = callbacks.DfuDataReceived;
	memset(&config, 0, sizeof(config));
	config.state = IQMESHSTATE_NOTCONFIGURED;
	memcpy(rdm_id, _rdm_id, 6);
	return true;
}

void iqMesh_extIrqInterupt(iqmesh_irq_edge_t edge)
{
	spiIqMesh_extIrqInterupt(edge);
}

void iqMesh_extSpiEvent(spi_event_t event, uint8_t *buffer, uint16_t length)
{
	spiIqMesh_extSpiEvent(event, buffer, length);
}

void iqMesh_setRdmId(const uint8_t *_rdm_id)
{
	memcpy(rdm_id, _rdm_id, 6);
	config.state = IQMESHSTATE_NOTCONFIGURED;
}

void iqMesh_unlinkIqData(void)
{
	unlink = true;
}

void iqMesh_startDfu_iqController(void)
{
	startDfu = true;
}
void iqMesh_setVisibility(iqmesh_visibility_t visibility)
{
	config.visibility    = visibility;
	config.visibilitySet = false;
}

bool iqMesh_isModuleStarted(void)
{
	return config.state == IQMESHSTATE_RUN;
}

uint32_t iqMesh_getModuleFirmwareVersion(void)
{
	return config.module_firmware_version;
}

uint32_t iqMesh_getModuleHardwareVersion(void)
{
	return config.module_hardware_version;
}

uint32_t iqMesh_getModuleBootloaderVersion(void)
{
	return config.module_bootloader_version;
}

const uint8_t *iqMesh_getModuleCpuId(void)
{
	return config.cpuId;
}

void iqMesh_getInformation(iqMesh_information_t *info)
{
	memcpy(info, &config.info, sizeof(iqMesh_information_t));
}

iqmesh_status_flags_t iqMesh_getStatusFlags()
{
	iqmesh_status_flags_t flags;
	memcpy(&flags, &config.status_flags, sizeof(iqmesh_status_flags_t));
	return flags;
}

bool iqMesh_writeRdm(const uint8_t *msg, uint16_t length)
{
	if(txRdm.set)
	{
		return false;
	}
	if(!AllocateBuffer(&txRdm, length + 1))
	{
		return false;
	}
	txRdm.buffer[0] = 0xFF;
	memcpy(&txRdm.buffer[1], msg, length);
	return true;
}
bool iqMesh_writeDmx(const uint8_t *dmx, uint16_t length)
{
	if(txDmx.set)
	{
		return false;
	}
	if(AllocateBuffer(&txDmx, length + 1))
	{
		return false;
	}
	txDmx.buffer[0] = 0xFF;
	memcpy(&txDmx.buffer[1], dmx, length);
	return true;
}

bool iqMesh_writeDfu(iqmesh_dfu_response response)
{
	if((!dfuState.started && !dfuState.finished) || dfuState.responseSet)
	{
		return false;
	}
	dfuState.dfuResponse = response;
	dfuState.responseSet = true;
	dfuState.finished    = false;
	return true;
}

bool iqMesh_task(uint32_t tickMs)
{
	bool todo;
	spi_message_header_t msg;
	uint8_t *bufPointer;

	if(!spiIqMesh_Task(tickMs))
	{
		return false;
	}

	bool irq = HardwareIqMeshGpioReadIrqPin();

	todo = (txRdm.set || txDmx.set || dfuState.responseSet || unlink || startDfu || rxRdm.set || rxDmx.set);

	switch(config.state)
	{
	case IQMESHSTATE_NOTCONFIGURED:
		// Send RDMID
		msg.cmd    = CMD_WRITE_REG | REG_RDM_ID;
		msg.length = REG_RDM_ID_LENGTH + 1;
		if(spiIqMesh_PrepareData(msg, &bufPointer))
		{
			bufPointer[0] = 0xFF;
			memcpy(&bufPointer[1], rdm_id, REG_RDM_ID_LENGTH);
			spiIqMesh_SetData(false);
		}
		break;
	case IQMESHSTATE_RDMIDSET:
		// Read RDM Id
		msg.cmd    = CMD_READ_REG | REG_RDM_ID;
		msg.length = REG_RDM_ID_LENGTH + 1;
		if(spiIqMesh_PrepareData(msg, &bufPointer))
		{
			bufPointer[0] = 0xFF;
			memset(&bufPointer[1], 0, REG_RDM_ID_LENGTH);
			spiIqMesh_SetData(false);
		}
		break;
	case IQMESHSTATE_RDMID_CHECK:
		// Check RDM ID if not correct go back to not configured
		for(uint8_t i = 0; i < 6; i++)
		{
			if(config.module_rdm_id[i] != rdm_id[i])
			{
				config.state = IQMESHSTATE_NOTCONFIGURED;
				return false;
			}
		}
		// Read Version
		msg.cmd    = CMD_READ_REG | REG_VERSION;
		msg.length = REG_VERSION_LENGTH + 1;
		if(spiIqMesh_PrepareData(msg, &bufPointer))
		{
			bufPointer[0] = 0xFF;
			memset(&bufPointer[1], 0, REG_VERSION_LENGTH);
			spiIqMesh_SetData(false);
		}
		break;
	case IQMESHSTATE_VERSIONRECEIVED:
		// Read CPUID
		msg.cmd    = CMD_READ_REG | REG_IQ_CPUID;
		msg.length = REG_IQ_CPUID_LENGTH + 1;
		if(spiIqMesh_PrepareData(msg, &bufPointer))
		{
			bufPointer[0] = 0xFF;
			memset(&bufPointer[1], 0, REG_IQ_CPUID_LENGTH);
			spiIqMesh_SetData(false);
		}
		break;
	case IQMESHSTATE_CPUIDRECEIVED:
		// Read NetworkInfo
		msg.cmd    = CMD_READ_REG | REG_NETWORK_INFORMATION;
		msg.length = REG_NETWORK_INFORMATION_LENGTH + 1;
		if(spiIqMesh_PrepareData(msg, &bufPointer))
		{
			bufPointer[0] = 0xFF;
			memset(&bufPointer[1], 0, REG_NETWORK_INFORMATION_LENGTH);
			spiIqMesh_SetData(false);
		}
		break;
	case IQMESHSTATE_NETWORKINFORECEIVED:
		// Read Status Flags
		msg.cmd    = CMD_READ_REG | REG_STATUS;
		msg.length = REG_STATUS_LENGTH + 1;
		if(spiIqMesh_PrepareData(msg, &bufPointer))
		{
			bufPointer[0] = 0xFF;
			memset(&bufPointer[1], 0, REG_STATUS_LENGTH);
			spiIqMesh_SetData(false);
		}
		break;
	case IQMESHSTATE_CHECKIRQ:
		if((config.irq & IRQ_SYS_RESTARTED) > 0)
		{
			msg.cmd    = CMD_WRITE_REG | REG_IRQ_FLAGS;
			msg.length = REG_IRQ_FLAGS_LENGTH + 1;
			if(spiIqMesh_PrepareData(msg, &bufPointer))
			{
				bufPointer[0] = 0xFF;
				bufPointer[1] = IRQ_SYS_RESTARTED;
				return spiIqMesh_SetData(false);
			}
			config.visibilitySet = false;
		}
		else
		{
			config.state = IQMESHSTATE_RUN;
		}
		break;
	case IQMESHSTATE_RUN:
		Run(irq);
		break;
	case IQMESHSTATE_RUNSET:
		config.state = IQMESHSTATE_UPDATEIRQ;
		break;
	case IQMESHSTATE_UPDATEIRQ:
		msg.cmd    = CMD_READ_REG | REG_IRQ_FLAGS;
		msg.length = REG_IRQ_FLAGS_LENGTH + 1;

		if(spiIqMesh_PrepareData(msg, &bufPointer))
		{
			bufPointer[0] = 0xFF;
			bufPointer[1] = 0;
			spiIqMesh_SetData(!irq);
		}
		break;
	}
	// Check if there is work todo
	return !todo;
}

bool iqMesh_isBusy(uint32_t tickMs)
{
	return !spiIqMesh_Task(tickMs);
}

// Internal Function
// Check irq and do stuff
static void Run(bool irq)
{
	// First Handle Tx Routines --------------------------
	// Handle Tx Rdm
	if(HandleTxRdm())
	{
		// transaction started leave function
		return;
	}
	// Handle Tx Dmx
	if(HandleTxDmx())
	{
		// transaction started leave function
		return;
	}
	// Handle Tx Dfu
	if(HandleTxDfu())
	{
		// transaction started leave function
		return;
	}

	// Handle Unlink
	if(HandleUnlink())
	{
		// transaction started leave function
		return;
	}

	// Handle Start Dfu
	if(HandleStartDfu())
	{
		// transaction started leave function
		return;
	}

	// Second handle Rx Callbacks
	if(CheckRxRdm())
	{
		// Rdm Callback triggered.
		// Return from function because normally there is a TxRdm handle
		return;
	}
	if(CheckRxDmx())
	{
		// Dmx Callback triggered
	}
	if(CheckRxDfu())
	{
		// Dfu Callback triggered
	}
	// Check IRQ Values and do stuff if necessary
	if(Handle_SysRestart())
	{
		// transaction started - leave function
		return;
	}

	if(HandleDfuFlag())
	{
		// transaction started leave function
		config.state = IQMESHSTATE_RUNSET;
		return;
	}

	if(HandleVisibility())
	{
		// transaction started - leave function
		return;
	}

	if(!irq)
	{
		if(HandleRxDmx())
		{
			// transaction started leave function
			config.state = IQMESHSTATE_RUNSET;
			return;
		}

		if(HandleRxRdm())
		{
			// transaction started leave function
			config.state = IQMESHSTATE_RUNSET;
			return;
		}

		if(HandleRfLink())
		{
			// transaction started leave function
			config.state = IQMESHSTATE_RUNSET;
			return;
		}
		if(HandleStatus())
		{
			// transaction started leave function
			config.state = IQMESHSTATE_RUNSET;
			return;
		}

		if(HandleDfuData())
		{
			// transaction started leave function
			config.state = IQMESHSTATE_RUNSET;
			return;
		}
	}

	if(HandleNfcFlag())
	{
		// transaction started leave function
		config.state = IQMESHSTATE_RUNSET;
		return;
	}
	if(!irq)
	{
		config.state = IQMESHSTATE_UPDATEIRQ;
	}
}

static void ReadRegister(uint8_t registerval, uint8_t *data, uint8_t length)
{
	switch(registerval)
	{
	case REG_STATUS:
		if(length >= REG_STATUS_LENGTH)
		{
			memcpy(&config.status_flags, data, 2);
		}
		break;
	case REG_IRQ_FLAGS:
		if(length >= REG_IRQ_FLAGS_LENGTH)
		{
			config.irq = data[0];
		}
		break;
	case REG_RDM_ID:
		if(length >= REG_RDM_ID_LENGTH)
		{
			memcpy(config.module_rdm_id, data, 6);
		}
		break;
	case REG_NETWORK_INFORMATION:
		if(length >= REG_NETWORK_INFORMATION_LENGTH)
		{
			memcpy(&config.info.uuid, data, 2);
			memcpy(&config.info.deviceCount, &data[2], 2);
			config.info.signalQuality = data[5];
			config.info.applicationId = data[6];
			memcpy(config.info.networkName, &data[7], 12);
			memcpy(config.info.networkIdentifier, &data[19], 3);
		}
		break;
	case REG_VERSION:
		if(length >= REG_VERSION_LENGTH)
		{
			memcpy(&config.module_firmware_version, data, 4);
			memcpy(&config.module_bootloader_version, &data[4], 4);
			memcpy(&config.module_hardware_version, &data[8], 4);
		}
		break;
	case REG_IQ_CPUID:
		if(length >= REG_IQ_CPUID_LENGTH)
		{
			memcpy(config.cpuId, data, 8);
		}
		break;
	}
}

static bool HandleTxRdm()
{
	if(txRdm.set)
	{
		spi_message_header_t msg;
		msg.cmd             = CMD_WRITE_RDM;
		msg.length          = txRdm.length;
		uint8_t *bufPointer = NULL;
		if(!spiIqMesh_PrepareData(msg, &bufPointer))
		{
			return false;
		}
		memcpy(bufPointer, txRdm.buffer, txRdm.length);
		if(spiIqMesh_SetData(false))
		{
			FreeBuffer(&txRdm);
			return true;
		}
		return false;
	}
	return false;
}
static bool HandleTxDmx()
{
	if(txDmx.set)
	{
		spi_message_header_t msg;
		msg.cmd             = CMD_WRITE_DMX;
		msg.length          = txDmx.length;
		uint8_t *bufPointer = NULL;
		if(!spiIqMesh_PrepareData(msg, &bufPointer))
		{
			return false;
		}
		memcpy(bufPointer, txDmx.buffer, txDmx.length);
		if(spiIqMesh_SetData(false))
		{
			FreeBuffer(&txDmx);
			return true;
		}
		return false;
	}
	return false;
}
static bool HandleTxDfu()
{
	if(dfuState.responseSet)
	{
		spi_message_header_t msg;
		msg.cmd             = CMD_WRITE_DFU;
		msg.length          = 2;  // NOP and Response
		uint8_t *bufPointer = NULL;
		if(!spiIqMesh_PrepareData(msg, &bufPointer))
		{
			return false;
		}
		bufPointer[0] = 0xFF;
		bufPointer[1] = dfuState.dfuResponse;
		if(spiIqMesh_SetData(false))
		{
			dfuState.responseSet = false;
			dfuState.dfuResponse = 0;
			return true;
		}
		return false;
	}
	return false;
}

static bool HandleUnlink()
{
	if(unlink)
	{
		spi_message_header_t msg;
		msg.cmd             = CMD_WRITE_REG | REG_STATUS;
		msg.length          = REG_STATUS_LENGTH;
		uint8_t *bufPointer = NULL;
		if(!spiIqMesh_PrepareData(msg, &bufPointer))
		{
			return false;
		}
		bufPointer[0] = 0xFF;
		bufPointer[1] = 1;
		if(spiIqMesh_SetData(false))
		{
			unlink = false;
			return true;
		}
		return false;
	}
	return false;
}
static bool HandleStartDfu()
{
	if(startDfu)
	{
		spi_message_header_t msg;
		msg.cmd             = CMD_WRITE_REG | REG_START_DFU;
		msg.length          = 2;
		uint8_t *bufPointer = NULL;
		if(!spiIqMesh_PrepareData(msg, &bufPointer))
		{
			return false;
		}
		bufPointer[0] = 0xFF;
		bufPointer[1] = 1;
		if(spiIqMesh_SetData(false))
		{
			startDfu = false;
			return true;
		}
		return false;
	}
	return false;
}

static bool Handle_SysRestart()
{
	if((config.irq & IRQ_SYS_RESTARTED) > 0)
	{
		spi_message_header_t msg;
		msg.cmd             = CMD_WRITE_REG | REG_IRQ_FLAGS;
		msg.length          = REG_IRQ_FLAGS_LENGTH + 1;
		uint8_t *bufPointer = NULL;
		if(!spiIqMesh_PrepareData(msg, &bufPointer))
		{
			return false;
		}
		bufPointer[0] = 0xFF;
		bufPointer[1] = IRQ_SYS_RESTARTED;
		config.state  = IQMESHSTATE_NOTCONFIGURED;
		return spiIqMesh_SetData(false);
	}
	return false;
}
static bool HandleVisibility()
{
	if(!config.visibilitySet)
	{
		spi_message_header_t msg;
		msg.cmd             = CMD_WRITE_REG | REG_SYSTEM_STATE;
		msg.length          = 1 + 1;
		uint8_t *bufPointer = NULL;
		if(!spiIqMesh_PrepareData(msg, &bufPointer))
		{
			return false;
		}
		bufPointer[0] = 0xFF;
		switch(config.visibility)
		{
		case IQMESH_VISIBILITY_IF_SELECTED_INPUT_SOURCE:
			bufPointer[1] = IQMESH_SYSTEM_NORMAL;
			break;
		case IQMESH_VISIBILITY_NON_ACTIVE_INPUT_SOURCE:
			bufPointer[1] = IQMESH_SYSTEM_OPEN_NON_ACTIVE_SOURCE;
			break;
		case IQMESH_VISIBILITY_ALWAYS:
			bufPointer[1] = IQMESH_SYSTEM_OPEN;
			break;
		}

		if(spiIqMesh_SetData(true))
		{
			config.visibilitySet = true;
			return true;
		}
	}
	return false;
}

static bool HandleRxRdm()
{
	if((config.irq & IRQ_RX_RDM) > 0)
	{
		spi_message_header_t msg;
		msg.cmd             = CMD_READ_RDM;
		msg.length          = RDM_FRAME_LENGTH;
		uint8_t *bufPointer = NULL;
		if(!spiIqMesh_PrepareData(msg, &bufPointer))
		{
			return false;
		}
		bufPointer[0] = 0xFF;
		memset(&bufPointer[1], 0, RDM_FRAME_LENGTH - 1);
		return spiIqMesh_SetData(true);
	}
	return false;
}
static bool HandleRxDmx()
{
	if((config.irq & IRQ_RX_DMX) > 0)
	{
		spi_message_header_t msg;
		uint8_t *bufPointer = NULL;
		msg.cmd             = CMD_READ_DMX;
		if(lastIqDmxMode == 255)
		{
			msg.length = iqSpecialDmxLength + 1;
		}
		else
		{
			msg.length = DMXLENGTH + 1;
		}
		if(!spiIqMesh_PrepareData(msg, &bufPointer))
		{
			return false;
		}
		// Fill Data
		bufPointer[0] = 0xFF;

		return spiIqMesh_SetData(true);
	}
	return false;
}
static bool HandleDfuFlag()
{
	if((config.irq & IRQ_DFU_FLAGS) > 0 && (config.irq & IRQ_DFU_FLAGS) != IRQ_DFU_FLAGS)
	{
		bool start = ((config.irq & IRQ_DFU_START) > 0);
		bool stop  = ((config.irq & IRQ_DFU_STOP) > 0);
		if(stop && dfuState.started)
		{
			FreeBuffer(&dfuState.rxDfu);
			dfuState.started  = false;
			dfuState.finished = true;
			if(DfuFlagReceivedHandler != NULL)
			{
				DfuFlagReceivedHandler(IQMESH_DFUFLAG_STOP);
			}
		}
		if(start && !dfuState.started)
		{
			if(dfuState.support)
			{
				AllocateBuffer(&dfuState.rxDfu, DFUPAGELENGTH + 2);
				dfuState.rxDfu.set = false;
				dfuState.started   = true;
				if(DfuFlagReceivedHandler != NULL)
				{
					DfuFlagReceivedHandler(IQMESH_DFUFLAG_START);
				}
			}
			else
			{
				// Write back thta dfu is not supported
				dfuState.dfuResponse = DFU_ERROR_NOT_IMPLEMENTED;
				dfuState.responseSet = true;
			}
		}
	}
	return false;
}

static bool HandleDfuData()
{
	if((config.irq & IRQ_DFU_FLAGS) > 0 && (config.irq & IRQ_DFU_FLAGS) != IRQ_DFU_FLAGS)
	{
		bool update = ((config.irq & IRQ_DFU_PACKET) > 0);
		if(update && dfuState.started && !dfuState.rxDfu.set)
		{
			spi_message_header_t msg;
			msg.cmd             = CMD_READ_DFU;
			msg.length          = 131;
			uint8_t *bufPointer = NULL;
			if(!spiIqMesh_PrepareData(msg, &bufPointer))
			{
				return false;
			}
			bufPointer[0] = 0xFF;
			memset(&bufPointer[1], 0, 131 - 1);
			return spiIqMesh_SetData(true);
		}
	}
	return false;
}
static bool HandleRfLink()
{
	if((config.irq & IRQ_RF_LINK) > 0)
	{
		spi_message_header_t msg;
		msg.cmd             = CMD_READ_REG | REG_NETWORK_INFORMATION;
		msg.length          = REG_NETWORK_INFORMATION_LENGTH + 1;
		statusFlag          = true;
		uint8_t *bufPointer = NULL;
		if(!spiIqMesh_PrepareData(msg, &bufPointer))
		{
			return false;
		}
		bufPointer[0] = 0xFF;
		return spiIqMesh_SetData(true);
	}
	return false;
}
static bool HandleStatus()
{
	if(statusFlag)
	{
		spi_message_header_t msg;
		msg.cmd             = CMD_READ_REG | REG_STATUS;
		msg.length          = REG_STATUS_LENGTH + 1;
		uint8_t *bufPointer = NULL;
		if(!spiIqMesh_PrepareData(msg, &bufPointer))
		{
			return false;
		}
		bufPointer[0] = 0xFF;
		statusFlag    = false;
		return spiIqMesh_SetData(true);
	}
	return false;
}
static bool HandleNfcFlag()
{
	if((config.irq & IRQ_NFC_WAKEUP) > 0)
	{
	}
	return false;
}

static bool CheckRxRdm()
{
	if(rxRdm.set)
	{
		if(RdmReceivedHandler != NULL)
		{
			RdmReceivedHandler(rxRdm.buffer, rxRdm.length);
		}
		FreeBuffer(&rxRdm);
		return true;
	}
	return false;
}
static bool CheckRxDmx()
{
	if(rxDmx.set)
	{
		if(lastIqDmxMode == rxDmx.buffer[0])
		{
			if(DmxReceivedHandler != NULL)
			{
				DmxReceivedHandler(rxDmx.buffer[0], &rxDmx.buffer[1], rxDmx.length - 1);
			}
		}
		lastIqDmxMode = rxDmx.buffer[0];
		FreeBuffer(&rxDmx);
		return true;
	}
	return false;
}
static bool CheckRxDfu()
{
	if(dfuState.started && dfuState.rxDfu.set)
	{
		if(DfuDataReceivedHandler != NULL)
		{
			uint16_t packetNr = (uint16_t)((dfuState.rxDfu.buffer[0] << 8) + (dfuState.rxDfu.buffer[1]));
			DfuDataReceivedHandler(packetNr, &dfuState.rxDfu.buffer[2], dfuState.rxDfu.length - 2);
		}
		dfuState.rxDfu.set = false;
		return true;
	}
	return false;
}

static bool AllocateBuffer(dynamic_buf_t *buf, uint16_t length)
{
	if(buf->buffer)
	{
		FreeBuffer(buf);
	}
	if(!buf->buffer)
	{
		buf->buffer = (uint8_t *)malloc(length);
		if(!buf->buffer)
		{
			return false;
		}
		else
		{
			buf->length = length;
			buf->set    = true;
		}
	}
	else
	{
		return false;
	}
	return true;
}

static void FreeBuffer(dynamic_buf_t *buf)
{
	if(buf->buffer)
	{
		free(buf->buffer);
		buf->buffer = NULL;
		buf->set    = false;
	}
}

// --------------- Callbacks ---------------------------------------------------

// First irq value before rx callback
static void spi_irq_callback_t(uint8_t irq)
{
	// Not used
	(void)irq;
}

// IRQ Flag
static void spi_irq_flag_callback_t(void)
{
	// Not used
}

static void spi_data_rx_callback_t(spi_message_header_t msg, uint8_t *buffer)
{
	// IRQ 0xFF is not valid - all flags occured at the same time is not possible. Throw frame away
	config.irq = buffer[0];
	if((msg.cmd & 0xC0) == CMD_READ_REG)
	{
		ReadRegister(msg.cmd & 0x3F, &buffer[1], (uint8_t)(msg.length - 1));
	}
	else if(msg.cmd == CMD_READ_DMX)
	{
		// Set Dmx Callback but via Task
		AllocateBuffer(&rxDmx, msg.length - 1);
		memcpy(rxDmx.buffer, &buffer[1], msg.length - 1);
	}
	else if(msg.cmd == CMD_READ_RDM)
	{
		// Set Rdm Callback but via Task
		AllocateBuffer(&rxRdm, msg.length - 1);
		memcpy(rxRdm.buffer, &buffer[1], msg.length - 1);
	}
	else if(msg.cmd == CMD_READ_DFU)
	{
		// Set Dfu Callback but via Task
		if((DFUPAGELENGTH + 2) >= msg.length - 1)
		{
			memcpy(dfuState.rxDfu.buffer, &buffer[1], msg.length - 1);
			dfuState.rxDfu.set = true;
		}
		else
		{
			// Should not occur
			return;
		}
	}
}

static void spi_finish_callback_t(bool success)
{
	if(!success)
	{
		return;
	}
	switch(config.state)
	{
	case IQMESHSTATE_NOTCONFIGURED:
		config.state = IQMESHSTATE_RDMIDSET;
		break;
	case IQMESHSTATE_RDMIDSET:
		config.state = IQMESHSTATE_RDMID_CHECK;
		break;
	case IQMESHSTATE_RDMID_CHECK:
		config.state = IQMESHSTATE_VERSIONRECEIVED;
		break;
	case IQMESHSTATE_VERSIONRECEIVED:
		config.state = IQMESHSTATE_CPUIDRECEIVED;
		break;
	case IQMESHSTATE_CPUIDRECEIVED:
		config.state = IQMESHSTATE_NETWORKINFORECEIVED;
		break;
	case IQMESHSTATE_NETWORKINFORECEIVED:
		config.state = IQMESHSTATE_CHECKIRQ;
		break;
	case IQMESHSTATE_CHECKIRQ:
		config.state = IQMESHSTATE_UPDATEIRQ;
		break;
	case IQMESHSTATE_RUN:
		break;
	case IQMESHSTATE_RUNSET:
		config.state = IQMESHSTATE_UPDATEIRQ;
		break;
	case IQMESHSTATE_UPDATEIRQ:
		config.state = IQMESHSTATE_RUN;
		break;
	}
}

static void spi_settxrx_callback_t(uint8_t *buf, uint16_t length)
{
	HardwareIqMeshSpiSetTxRx(buf, length);
}
static void spi_setcs_callback_t(bool set)
{
	HardwareIqMeshSpiSetCsPin(set);
}
