/**
 ******************************************************************************
 * @file           : button.h
 * @brief          : Simple example to implement a button with short / long press
 ******************************************************************************
 * @attention
 *
 * Copyright (c) Kuenzler Technologies GbR
 * All rights reserved.
 *
 *
 ******************************************************************************
 */

#ifndef INC_BUTTON_H_
#define INC_BUTTON_H_

#include "stdbool.h"
#include "stdint.h"

typedef struct
{
	void (*ButtonShortPress)(void);
	void (*ButtonLongPress)(uint8_t count);
	void (*ButtonDoubleClick)(void);
	void (*ButtonPressed)(void);
	void (*ButtonReleased)(uint8_t count);
} button_callbacks_t;

typedef enum {
	StateIdle,
	StateDebounce,
	StatePressed,
	StateClickUp,
	StateClickIdle,
	StateDoubleClickDebounce,
	StateDoubleClick,
	StateLongClickCount,
	StateOtherUp,
} button_state_t;

typedef struct
{
	bool eventButtonPressReleased;
	bool eventButtonPressed;
	bool eventButtonDoubleClick;
	bool eventButtonLongPress;
	bool eventButtonShortPress;
} button_flags_t;

typedef struct
{
	button_state_t btnState;
	uint16_t longPressTimeMs;
	uint16_t longPressCounter;
	bool activeStatePressed;
	bool useTask;
	button_callbacks_t handler;
	button_flags_t flags;
	uint8_t id;
} button_instance_t;

void InitButton(button_instance_t *instance, uint8_t instanceId, button_callbacks_t callbacks,
				uint16_t _longPressTimeMs, bool _activeStatePressed, bool useTask);
void ButtonTask(button_instance_t *instance);
void ButtonInteruptIrq(button_instance_t *instance, bool pressed);
void ButtonWaitTimeElapsed(button_instance_t *instance);

/*==============================================================================*/
/* External defined hardware functions                                        	*/
/*==============================================================================*/
// Stop and Start Timer with comitted time and call ButtonWaitTimeElapsed after elapsing
extern void HardwareButtonTimerStopStartWaitTime(uint16_t timeMs, uint8_t instanceId);
extern bool HardwareButtonGpioReadPin(uint8_t instanceId);  // Return State of the button
#endif                                                      /* INC_BUTTON_H_ */
