/**
  ******************************************************************************
  * @file           : dfu_update.c
  * @brief          : Simple example for showing dfu update functionality
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Kuenzler Technologies GbR
  * All rights reserved.
  *
  *
  ******************************************************************************
*/

/*==============================================================================*/
/* Includes                                                                  	*/
/*==============================================================================*/
#include "dfu_update.h"
#include <string.h>
#include "libIqMeshSpiInterface.h"
#include "stdlib.h"

/*==============================================================================*/
/* Defines & Macros                                                           	*/
/*==============================================================================*/

#define SCRDM 0xCC
#define SCSUBMESSAGE 0x1
#define RDMHEADERLENGTH 24

#define RDMCPID_IQ  0xA000

#define RDMCOMMAND_GET  0x20
#define RDMCOMMAND_GET_RESPONSE  0x21
#define RDMCOMMAND_SET  0x30
#define RDMCOMMAND_SET_RESPONSE  0x31

#define RDM_ACK 0
#define RDM_NACK_REASON 2

#define RDM_NR_UNKNOWN_PID 0

static const uint8_t ciq_response[66] =
{
	1,//uint8_t protocollVersion;
	BOOTLOADER_RDM_MODEL_ID >> 8, (uint8_t)BOOTLOADER_RDM_MODEL_ID,//uint16_t deviceModelId
	0,0,0,0,0,0,0,0,//uint64_t serial
	0,0,0,0,//uint8_t iqControllerFirmware[4] -> not necessary
	0,0,0,0,//uint8_t iqControllerBootloaderFirmware[4]
	0,0,0,0,//uint8_t fixtureFirmware[4]
	BOOTLOADER_VERSION_MAJOR,BOOTLOADER_VERSION_MINOR,BOOTLOADER_VERSION_PATCH,BOOTLOADER_VERSION_RELEASE,//uint8_t fixtureBootloaderFirmware[4]
	0,0,0,0,0,0,0,0,//uint8_t iqControllerCpuId[8]
	0,0,0,0,0,0,0,0,0,0,0,0,//uint8_t fixtureMainCpuId[12]
	0,//uint8_t battery - no battery available
	2,//uint8_t powerSupplyState
	0,//uint8_t reserved
	0,//uint8_t inputSource
	0,//uint8_t inputState
	0,//uint8_t inputQuality
	0,1,//uint16_t dmxStartaddress
	1,//uint8_t dmxPersonality
	0,0,//uint16_t dmxFootprint
	0,//uint8_t errorCount
	0,1,//uint16_t supportedInputSource
	0,//uint8_t featureIqSleep_old
	0,//uint8_t featureShippingMode
	0,//uint8_t featureEmergency
	0,//uint8_t featureRuntimeSelection
	0,//uint8_t featureAntitheftMode
};

typedef struct
{
  uint16_t pid_num;
  uint8_t pdl;
  uint8_t targetRdmId[6];
  uint8_t sourceRdmId[6];
  uint8_t tn;
  uint8_t cc;
} rdm_message_header_t;

/* Timeout if no package is transfered - 如果没有传输包则超时 */
#define DFU_START_TIMEOUT_MS	120000
#define DFU_PACKET_TIMEOUT_MS	30000
/* Time to wait before restart iQ.Controller. Make sure that success response reached smartphone - 重新启动 iQ.Controller 之前等待的时间。确保成功响应到达智能手机 */
#define DFU_FINALIZE_MS	5000

/*==============================================================================*/
/* Public variables                                                           	*/
/*==============================================================================*/

/*==============================================================================*/
/* Private variables                                                          	*/
/*==============================================================================*/

/* DFU State - DFU 状态 */
static dfu_state_t dfuActualState = DFU_STATE_IDLE;
static iqmesh_dfu_response responseToController = DFU_SUCCESS;

/* Firmware Size that we have received - 我们收到的固件大小 */
static volatile uint32_t dfuFwWrittenSize;
static uint16_t dfuPageNr;

/* Configuration written in flash - 配置写入闪存*/
static dfu_config_t *cfgFlash   = (dfu_config_t*) (FLASH_ADDR_CONFIG);
static uint32_t startTime = 0;
static uint8_t alignementBuffer[USER_FLASH_BYTE_ALIGNMENT];
static uint8_t alignementBufferFillCount = 0;

/*==============================================================================*/
/* Private functions                                                          	*/
/*==============================================================================*/
static bool dfu_check_app(void);
static bool dfu_clear_request(void);
static void dfu_goto_application( void );
static void PrepareDfu(void);
static bool DfuCheckCustomHeader(const uint8_t* data, uint16_t length);
static bool DfuFinalize(void);
// Helper
static bool DecodeRDMMessage(rdm_message_header_t* header, const uint8_t* msg,uint16_t length);
static void EncodeRdmMessage(rdm_message_header_t msg,uint8_t* data, uint8_t* buf, bool ack);
static uint32_t CalculateCRC(uint8_t* data, uint32_t length);
static bool DfuReceiveData(const uint8_t* data, uint16_t length);

// Callbacks
static void DmxReceivedCallback(uint8_t personality, const uint8_t* buf, uint16_t length);
static void RdmReceivedCallback(const uint8_t* buf, uint16_t length);
static void DfuFlagReceivedCallback(iqmesh_dfu_flag_t flag);
static void DfuDataReceivedCallback(uint16_t packetNr,const uint8_t* buf, uint16_t length);

/*==============================================================================*/
/* Function definitions                                                     	*/
/*==============================================================================*/
bool DfuCheckBootReason()
{
	//Read the reboot cause and act accordingly - 读�?��?�?�原因并采�?�相应措施
	dfu_config_t* cfg = (dfu_config_t*)FLASH_ADDR_CONFIG;

	bool   goto_dfu_mode = false;
	bool timeout = false;

	switch(cfg->reboot_cause)
	{
		case BOOT_NORMAL:
		{
		/*
		 * It is a normal boot. So, do nothing here. 这是正常的�?�动。所以，�?�?在这里�?�任何事情。
		 */
		break;
		}
		case BOOT_DFU_REQUEST:
		/*
		 * Application has requested for the DFU update - 应用程�?已请求 DFU 更新
		 */
		dfu_clear_request();
		goto_dfu_mode = true;
		timeout = true;
		break;
		case BOOT_FIRST_TIME:
		{
			/*
			 * This is the first time boot. Directly go to DFU mode.
			 * 这是第一次�?�动。直接进入DFU模�?。
			 */
			goto_dfu_mode = true;
			timeout = false;
			break;
		}
		default:
			/* should not get here - �?应该到达这里 */
			break;
	}
	if(HardwareCheckDFUPin())
	{
		goto_dfu_mode = true;
		timeout = true;
	}

	/*Start the Firmware or Application update or check the already flashed app
	 * - �?�动固件或应用程�?更新或检查已刷新的应用程�?*/
	if(!goto_dfu_mode)
	{
		/* Check application, if it is available - 检查应用程�?是�?��?�用*/
		if(dfu_check_app())
		{
			// Jump to application
			dfu_goto_application();
		}
		else
		{
			/* application is not valid. Wait for DFU Update - 申请无效。等待DFU更新*/
			timeout = false;
		}
	}
	return timeout;
}

dfu_response_t DfuDownloadFlash(bool timeout)
{
	dfu_response_t response = DFU_OK;
	uint32_t tempTime = HardwareTimeMs();
	iqmesh_callbacks_t callbacks;
	dfuActualState         = DFU_STATE_IDLE;
	startTime 		  = HardwareTimeMs();

	/* Initialize SPI iQ.Mesh Library - 初始化 SPI iQ.Mesh 库 */
	callbacks.DfuDataReceived = &DfuDataReceivedCallback;
	callbacks.DfuFlagReceived = &DfuFlagReceivedCallback;
	callbacks.DmxReceived = &DmxReceivedCallback;
	callbacks.RdmReceived = &RdmReceivedCallback;
	
	// Modify rdm_id same as the application if possible
	iqMesh_init((const uint8_t*)rdm_id, 100,true, callbacks);

	/* Enable iQ.Controller - 启用 iQ.Controller */
	HardwareGpioSetFixtureIsOnPin(true);

	// Signal Bootloader Ready
	HardwareGpioSetSignalLed(true);

	while (1)
	{
		/* Returns true if there is nothing todo - 如果没有任何待办事项，则返回 true*/
	  if(iqMesh_task(HardwareTimeMs()))
	  {

		  /* General Timeout  -  一般超时*/
		  if(dfuActualState == DFU_STATE_IDLE && timeout && (startTime + DFU_START_TIMEOUT_MS) < HardwareTimeMs())
		  {
			  /* Timed out. This can happen if the application requested a DFU, but there is no transfer - 时间到。如果应用程序请求 DFU，但没有传输，则可能会发生这种情况 */
			  response = DFU_TIMEOUT;
			  break;
		  }
		  /* DFU Packet Timeout  -  DFU 数据包超时*/
		  if((dfuActualState == DFU_STATE_DATA) && (startTime + DFU_PACKET_TIMEOUT_MS) < HardwareTimeMs())
		  {
			  /* Timed out. This can happen if the application started a DFU Transfer, but interupted the transfer without aborting - 超时了。如果应用程序启动了 DFU 传输，但在没有中止的情况下中断了传输，则可能会发生这种情况 */
			  response = DFU_TIMEOUT;
			  break;
		  }
		  /* Timeout after packet - 数据包发送后超时*/
		  if(dfuActualState == DFU_STATE_START && (startTime + DFU_PACKET_TIMEOUT_MS) < HardwareTimeMs())
		  {
			  /* Timed out */
			  response =  DFU_TIMEOUT;
			  break;
		  }
		  if(dfuActualState == DFU_STATE_FINISH)
		  {
			  if(DfuFinalize())
			  {
				  response = DFU_OK;
				  responseToController = DFU_SUCCESS;
				  iqMesh_writeDfu(DFU_SUCCESS);
			  }
			  else
			  {
				  response = DFU_ERR;
				  responseToController = DFU_ERROR_CORRUPTED_IMAGE;
				  iqMesh_writeDfu(DFU_ERROR_CORRUPTED_IMAGE);
			  }
			  break;
		  }
		  else if(dfuActualState == DFU_STATE_ABORT)
		  {
			  response =  DFU_ERR;
			  //iqMesh_writeDfu(DFU_ERROR_CORRUPTED_IMAGE);
			  break;
		  }
	  }

	  if(tempTime + 2000 <= HardwareTimeMs())
	  {
		  tempTime = HardwareTimeMs();
	  }
	}

	/* Give the iQ.Controller time to process in case the fixture keeps the iQ.Controller alive -  为 iQ.Controller 提供处理时间，以防夹具使 iQ.Controller 保持活动状态*/
	startTime = HardwareTimeMs();
	while((startTime + DFU_FINALIZE_MS) > HardwareTimeMs())
	{
		iqMesh_task(HardwareTimeMs());
	}

	/* Set iQ.Controller to sleep because this controller will restart anyway - 将 iQ.Controller 设置为睡眠状态，因为该控制器无论如何都会重新启动*/
	HardwareGpioSetFixtureIsOnPin(false);

  return response;
}

void DfuIqMeshSpiTxRxComplete(uint8_t* data, uint16_t length)
{
	iqMesh_extSpiEvent(SPI_EVENT_COMPLETION, data, length);
}

void DfuIqMeshIrqPinIrq(bool set)
{
	if(set)
	{
		iqMesh_extIrqInterupt(SPIEXT_IRQ_EDGE_RISING);
	}
	else
	{
		iqMesh_extIrqInterupt(SPIEXT_IRQ_EDGE_FALLING);
	}
}
/*==============================================================================*/
/* Private Functions                                                     	*/
/*==============================================================================*/
static bool dfu_clear_request()
{
	// Calculate aligned length for compatible flash layout
	uint16_t alignedLength = sizeof(dfu_config_t) / USER_FLASH_BYTE_ALIGNMENT;
	if(sizeof(dfu_config_t) % USER_FLASH_BYTE_ALIGNMENT > 0)
	{
		alignedLength += USER_FLASH_BYTE_ALIGNMENT;
	}
	// Reserve buffer for any alignemnt
	uint8_t* writeBuffer = malloc(alignedLength * USER_FLASH_BYTE_ALIGNMENT);
	if(writeBuffer == NULL)
	{
		return false;
	}
	memcpy(writeBuffer,cfgFlash, sizeof(dfu_config_t));
	/* Set Signature */
	dfu_config_t* config = (dfu_config_t*)writeBuffer;
	config->reboot_cause = BOOT_NORMAL;
	config->bootloaderVersion[0] = BOOTLOADER_VERSION_MAJOR;
	config->bootloaderVersion[1] = BOOTLOADER_VERSION_MINOR;
	config->bootloaderVersion[2] = BOOTLOADER_VERSION_PATCH;
	config->bootloaderVersion[3] = BOOTLOADER_VERSION_RELEASE;
	/* Pages needed for saving the settings - 保存设置所需的页面*/
	if(!HardwareEraseFlash(FLASH_ADDR_CONFIG, FLASH_PAGE_CONFIG, 1))
	{
		free(writeBuffer);
		return false;
	}
	if(!HardwareWriteFlash(FLASH_ADDR_CONFIG, alignedLength, writeBuffer))
	{
		free(writeBuffer);
		return false;
	}
	free(writeBuffer);
	return true;
}

static bool dfu_check_app()
{
	uint32_t fileSize = 0;
	uint32_t fileCrc = 0;
	uint32_t cal_data_crc = 0;
	memcpy((uint8_t*)&fileCrc, (uint8_t*)&(cfgFlash->file.fileCRC), 4);
	memcpy((uint8_t*)&fileSize, (uint8_t*)&(cfgFlash->file.fileSize), 4);
	if(fileSize >= FLASH_APP_LENGTH || fileSize == 0)
	{
		return false;
	}

	cal_data_crc = CalculateCRC((uint8_t*)FLASH_ADDR_APP, fileSize);
	if(cal_data_crc != fileCrc)
	{
		return false;
	}
	return true;
}

static void dfu_goto_application(void)
{
	uint32_t appStack = (uint32_t) *((volatile uint32_t *) FLASH_ADDR_APP);

	//Get Program Count -> ResetHandler
	/* Get the application entry point (Second entry in the application vector table)*/
	void (*app_reset_handler)(void) = (void*)(*((volatile uint32_t*) (FLASH_ADDR_APP + 4U)));

	HardwareSetMsp(appStack);
	/* Start the application */
	app_reset_handler();
}

static void PrepareDfu()
{
	/* Reset the variables  - 重置变量*/
	dfuFwWrittenSize = 0u;
	dfuPageNr          = 0u;
	dfuActualState = DFU_STATE_START;
	alignementBufferFillCount = 0;
	/* Set time that no timeout occurs - 设置不超时的时间*/
	startTime = HardwareTimeMs();
}

static bool DfuCheckCustomHeader(const uint8_t* msg, uint16_t length)
{
	if(length < sizeof(dfu_file_header_t))
	{
		return false;
	}

	uint16_t alignedLength = sizeof(dfu_config_t) / USER_FLASH_BYTE_ALIGNMENT;
	if(sizeof(dfu_config_t) % USER_FLASH_BYTE_ALIGNMENT > 0)
	{
		alignedLength += USER_FLASH_BYTE_ALIGNMENT;
	}
	// Reserve buffer for any alignemnt
	uint8_t* writeBufferDfuConfig = malloc(alignedLength * USER_FLASH_BYTE_ALIGNMENT);
	if(writeBufferDfuConfig == NULL)
	{
		return false;
	}
	/* Write the configuration - 编写配置*/
	dfu_config_t* config = (dfu_config_t*)writeBufferDfuConfig;
	config->reboot_cause = BOOT_NORMAL;
	config->bootloaderVersion[0] = BOOTLOADER_VERSION_MAJOR;
	config->bootloaderVersion[1] = BOOTLOADER_VERSION_MINOR;
	config->bootloaderVersion[2] = BOOTLOADER_VERSION_PATCH;
	config->bootloaderVersion[3] = BOOTLOADER_VERSION_RELEASE;
	memcpy((uint8_t*)&(config->file),msg,sizeof(dfu_file_header_t));
	/* Check Signature - 检查签名*/
	for(uint8_t i=0;i< sizeof(SIGNATURE);i++)
	{
		if(SIGNATURE[i] != config->file.signature[i])
		{
			free(writeBufferDfuConfig);
			return false;
		}
	}
	/* Pages needed for Application - 申请所需页面*/
	uint8_t pages = (uint8_t)( (config->file.fileSize + (FLASH_ADDR_APP - FLASH_ADDR_CONFIG)) / FLASHPAGE_SIZE);
	if((config->file.fileSize + (FLASH_ADDR_APP - FLASH_ADDR_CONFIG)) % FLASHPAGE_SIZE > 0)
	{
		pages++;
	}
	 /* Pages needed for Application + one Page for saving the settings - 应用程序所需的页面 + 保存设置的页面 */
	/* Attention. You may have to modify the page count depending on the adress. - 注意力。您可能需要根據地址修改頁數。*/
	if(!HardwareEraseFlash(FLASH_ADDR_CONFIG, FLASH_PAGE_CONFIG, pages))
	{
		free(writeBufferDfuConfig);
		return false;
	}

	// Write Configuration
	if(!HardwareWriteFlash(FLASH_ADDR_CONFIG, alignedLength, writeBufferDfuConfig))
	{
		free(writeBufferDfuConfig);
		return false;
	}
	free(writeBufferDfuConfig);

	alignedLength = (length - sizeof(dfu_file_header_t)) / USER_FLASH_BYTE_ALIGNMENT;
	uint8_t rest = (length - sizeof(dfu_file_header_t)) % USER_FLASH_BYTE_ALIGNMENT;
	if(rest > 0)
	{
		// Not possible to write into flash right now. We have to buffer it when the next data arrives
		memcpy(alignementBuffer, &msg[length - rest -1], rest);
		alignementBufferFillCount = rest;
	}
	uint8_t* writeBufferData = malloc(alignedLength * USER_FLASH_BYTE_ALIGNMENT);
	memcpy(writeBufferData, &msg[sizeof(dfu_file_header_t)], alignedLength * USER_FLASH_BYTE_ALIGNMENT);

	dfuFwWrittenSize += alignedLength * USER_FLASH_BYTE_ALIGNMENT;
	if(!HardwareWriteFlash(FLASH_ADDR_APP, alignedLength, writeBufferData))
	{
		free(writeBufferData);
		return false;
	}
	free(writeBufferData);
	return true;
}

static bool DfuReceiveData(const uint8_t* data, uint16_t length)
{
	uint16_t alignedLength = (length + alignementBufferFillCount) / USER_FLASH_BYTE_ALIGNMENT;
	uint8_t* writeBufferData = malloc(alignedLength * USER_FLASH_BYTE_ALIGNMENT);
	uint8_t rest = (length + alignementBufferFillCount) % USER_FLASH_BYTE_ALIGNMENT;
	uint8_t startIndex = 0;
	if(alignementBufferFillCount > 0)
	{
		memcpy(writeBufferData,alignementBuffer,alignementBufferFillCount);
		startIndex = alignementBufferFillCount;
		alignementBufferFillCount = 0;
	}
	if(rest > 0)
	{
		memcpy(alignementBuffer, &data[length - rest -1], rest);
		alignementBufferFillCount = rest;
	}
	memcpy(&writeBufferData[startIndex], data, (alignedLength *USER_FLASH_BYTE_ALIGNMENT) - startIndex );

	if(!HardwareWriteFlash(FLASH_ADDR_APP + dfuFwWrittenSize, alignedLength, writeBufferData))
	{
		free(writeBufferData);
		return false;
	}
	dfuFwWrittenSize += alignedLength * USER_FLASH_BYTE_ALIGNMENT;
	free(writeBufferData);
	return true;
}

static bool DfuFinalize()
{
	/* CHeck if there is unaligned data in buffer */
	if(alignementBufferFillCount > 0)
	{
		uint8_t* writeBufferData = malloc(USER_FLASH_BYTE_ALIGNMENT);
		memcpy(writeBufferData,alignementBuffer, alignementBufferFillCount);
		if(!HardwareWriteFlash(FLASH_ADDR_APP + dfuFwWrittenSize, 1, writeBufferData))
		{
			free(writeBufferData);
			return false;
		}
		dfuFwWrittenSize += USER_FLASH_BYTE_ALIGNMENT;
		alignementBufferFillCount = 0;
		free(writeBufferData);
	}
	/* Check transfered size is bigger or as big as file size  - 检查传输的大小是否大于或与文件大小一样大*/
	uint32_t fileSize = 0;
	memcpy((uint8_t*)&fileSize, (uint8_t*)&(cfgFlash->file.fileSize), 4);
	if(fileSize > dfuFwWrittenSize)
	{
		return false;
	}
	return dfu_check_app();
}

/*==============================================================================*/
/* Callback Functions                                                     	*/
/*==============================================================================*/

static void DmxReceivedCallback(uint8_t personality, const uint8_t* buf, uint16_t length)
{
	/* DMX Data not handled - DMX 数据未处理*/
	(void) personality;
	(void) buf;
	(void)length;
}
static void RdmReceivedCallback(const uint8_t* buf, uint16_t length)
{
	/* Handle necessary RDM Data - 处理必要的 RDM 数据*/
	rdm_message_header_t header;
	if(!DecodeRDMMessage(&header, buf,length))
	{
		return;
	}
	if(header.targetRdmId[0] != rdm_id[0] ||
		header.targetRdmId[1] != rdm_id[1] ||
		header.targetRdmId[2] != rdm_id[2] ||
		header.targetRdmId[3] != rdm_id[3] ||
		header.targetRdmId[4] != rdm_id[4] ||
		header.targetRdmId[5] != rdm_id[5])
	{
		return;
	}
	/* Only Rdm iQ command is necessary in DFU - DFU 中仅需要 Rdm IQ 命令 */
    if(header.pid_num == RDMCPID_IQ && header.cc == RDMCOMMAND_GET) // 0xA000
	{
		uint8_t msg[RDMHEADERLENGTH + 2 + sizeof(ciq_response)];
		header.pdl = sizeof(ciq_response);
		EncodeRdmMessage(header,(uint8_t*)ciq_response,msg,true);
		iqMesh_writeRdm(msg,RDMHEADERLENGTH + 2 + sizeof(ciq_response));
	}
    else /* Send Nack Response - 发送 Nack 响应 */
    {
    	uint8_t msg[RDMHEADERLENGTH + 2 + 2];
		header.pdl = 2;
		uint16_t tempReason = RDM_NR_UNKNOWN_PID;
		EncodeRdmMessage(header,(uint8_t*)&tempReason,msg,false);
		iqMesh_writeRdm(msg,RDMHEADERLENGTH + 2 + 2);
    }
}

static void DfuFlagReceivedCallback(iqmesh_dfu_flag_t flag)
{
	/* Handle DFU start/stop flags - 处理 DFU 启动/停止标志*/
	if(flag == IQMESH_DFUFLAG_STOP)
	{
		dfuActualState = DFU_STATE_FINISH;
	}

	if(flag == IQMESH_DFUFLAG_START)
	{
		/* Prepare for Update - 准备更新*/
		PrepareDfu();
		/* Answer DFU with Success in order to acknowledge DFU - 回答 DFU 成功以确认 DFU*/
		iqMesh_writeDfu(DFU_SUCCESS);
	}
}
static void DfuDataReceivedCallback(uint16_t packetNr,const uint8_t* buf, uint16_t length)
{
	if(dfuActualState == DFU_STATE_START )
	{
		/* Check header if file is correct - 检查文件头是否正确 */
		if(DfuCheckCustomHeader(buf,length))
		{
			dfuActualState = DFU_STATE_DATA;
			dfuPageNr = packetNr;
			return;
		}
		else
		{
			/* Error occured and header is incorrect - 发生错误并且标头不正确 */
			iqMesh_writeDfu(DFU_ERROR_CORRUPTED_HEADER);
			responseToController = DFU_ERROR_CORRUPTED_HEADER;
			dfuActualState = DFU_STATE_ABORT;
			return;
		}
	}
	else if(dfuActualState == DFU_STATE_DATA)
	{
		if(packetNr == (dfuPageNr+1))
		{
			/* Set back timeout timer after packet arrived - 数据包到达后设置超时计时器 */
			startTime = HardwareTimeMs();
			if(!DfuReceiveData(buf,length))
			{
				/* Error occured while writing data - 写入数据时发生错误 */
				iqMesh_writeDfu(DFU_ERROR_CORRUPTED_IMAGE);
				responseToController = DFU_ERROR_UNKNOWN;
				dfuActualState = DFU_STATE_ABORT;
				return;
			}
			else
			{
				dfuPageNr++;
				return;
			}
		}
		else
		{
			/* Error occured wrong packet number received - 发生错误，收到错误的数据包编号*/
			iqMesh_writeDfu(DFU_ERROR_WRONG_PACKETNR);
			responseToController = DFU_ERROR_WRONG_PACKETNR;
			dfuActualState = DFU_STATE_ABORT;
		}

	}
}


static void EncodeRdmMessage(rdm_message_header_t msg,uint8_t* data, uint8_t* buf, bool ack)
{
    buf[0] = (SCRDM);
    buf[1] = (SCSUBMESSAGE);
    buf[2] = ((uint8_t)(RDMHEADERLENGTH + msg.pdl));
    buf[3] =  msg.sourceRdmId[0];
    buf[4] =  msg.sourceRdmId[1];
    buf[5] =  msg.sourceRdmId[2];
    buf[6] =  msg.sourceRdmId[3];
    buf[7] =  msg.sourceRdmId[4];
    buf[8] =  msg.sourceRdmId[5];


    buf[9] =  msg.targetRdmId[0];
    buf[10] =  msg.targetRdmId[1];
    buf[11] =  msg.targetRdmId[2];
    buf[12] =  msg.targetRdmId[3];
    buf[13] =  msg.targetRdmId[4];
    buf[14] =  msg.targetRdmId[5];
    buf[15] = msg.tn;
    buf[16] = (1); // Port ID
    buf[17] = (0); // Message Count
    buf[18] = (0); // Subdevice Field
    buf[19] = (0); // Subdevice Field
    if(msg.cc == RDMCOMMAND_GET)
	{
	buf[20] = RDMCOMMAND_GET_RESPONSE;
	}
	else if(msg.cc == RDMCOMMAND_SET)
	{
	  buf[20] = RDMCOMMAND_SET_RESPONSE;
	}
    buf[21] = ((msg.pid_num >> 8) & 0xFF);
    buf[22] = ((msg.pid_num >> 0) & 0xFF);
    buf[23] = (msg.pdl);

    /* Set Response type if response message */
    if(ack)
    {
    	buf[16] = RDM_ACK;
    }
    else
    {
    	buf[16] = RDM_NACK_REASON;
    }

    memcpy(&buf[24],data,msg.pdl);
    uint16_t checksum = 0;
    for(uint8_t i= 0; i< 24 + msg.pdl; i++)
    {
      checksum += buf[i];
    }
    buf[24 + msg.pdl] =((checksum >> 8) & 0xFF);
    buf[25 + msg.pdl] =((checksum >> 0) & 0xFF);
}

static bool DecodeRDMMessage(rdm_message_header_t* header, const uint8_t* msg, uint16_t length)
{
	// Short Check Length ,Header, make sure it is get command
	if(length <= RDMHEADERLENGTH || msg[0] != SCRDM || msg[1] !=SCSUBMESSAGE || msg[2] +1 > length)
	{
	  return false;
	}
	header->pid_num = (uint16_t)(msg[21] << 8) + msg[22];
	header->targetRdmId[0] = msg[3];
	header->targetRdmId[1] = msg[4];
	header->targetRdmId[2] = msg[5];
	header->targetRdmId[3] = msg[6];
	header->targetRdmId[4] = msg[7];
	header->targetRdmId[5] = msg[8];
	header->sourceRdmId[0] = msg[9];
	header->sourceRdmId[1] = msg[10];
	header->sourceRdmId[2] = msg[11];
	header->sourceRdmId[3] = msg[12];
	header->sourceRdmId[4] = msg[13];
	header->sourceRdmId[5] = msg[14];
	header->tn =  msg[15];
	header->cc = msg[20];
	return true;
}


static const uint32_t crc32_tab[] = {
        0x00000000, 0x77073096, 0xee0e612c, 0x990951ba, 0x076dc419, 0x706af48f,
        0xe963a535, 0x9e6495a3, 0x0edb8832, 0x79dcb8a4, 0xe0d5e91e, 0x97d2d988,
        0x09b64c2b, 0x7eb17cbd, 0xe7b82d07, 0x90bf1d91, 0x1db71064, 0x6ab020f2,
        0xf3b97148, 0x84be41de, 0x1adad47d, 0x6ddde4eb, 0xf4d4b551, 0x83d385c7,
        0x136c9856, 0x646ba8c0, 0xfd62f97a, 0x8a65c9ec, 0x14015c4f, 0x63066cd9,
        0xfa0f3d63, 0x8d080df5, 0x3b6e20c8, 0x4c69105e, 0xd56041e4, 0xa2677172,
        0x3c03e4d1, 0x4b04d447, 0xd20d85fd, 0xa50ab56b, 0x35b5a8fa, 0x42b2986c,
        0xdbbbc9d6, 0xacbcf940, 0x32d86ce3, 0x45df5c75, 0xdcd60dcf, 0xabd13d59,
        0x26d930ac, 0x51de003a, 0xc8d75180, 0xbfd06116, 0x21b4f4b5, 0x56b3c423,
        0xcfba9599, 0xb8bda50f, 0x2802b89e, 0x5f058808, 0xc60cd9b2, 0xb10be924,
        0x2f6f7c87, 0x58684c11, 0xc1611dab, 0xb6662d3d, 0x76dc4190, 0x01db7106,
        0x98d220bc, 0xefd5102a, 0x71b18589, 0x06b6b51f, 0x9fbfe4a5, 0xe8b8d433,
        0x7807c9a2, 0x0f00f934, 0x9609a88e, 0xe10e9818, 0x7f6a0dbb, 0x086d3d2d,
        0x91646c97, 0xe6635c01, 0x6b6b51f4, 0x1c6c6162, 0x856530d8, 0xf262004e,
        0x6c0695ed, 0x1b01a57b, 0x8208f4c1, 0xf50fc457, 0x65b0d9c6, 0x12b7e950,
        0x8bbeb8ea, 0xfcb9887c, 0x62dd1ddf, 0x15da2d49, 0x8cd37cf3, 0xfbd44c65,
        0x4db26158, 0x3ab551ce, 0xa3bc0074, 0xd4bb30e2, 0x4adfa541, 0x3dd895d7,
        0xa4d1c46d, 0xd3d6f4fb, 0x4369e96a, 0x346ed9fc, 0xad678846, 0xda60b8d0,
        0x44042d73, 0x33031de5, 0xaa0a4c5f, 0xdd0d7cc9, 0x5005713c, 0x270241aa,
        0xbe0b1010, 0xc90c2086, 0x5768b525, 0x206f85b3, 0xb966d409, 0xce61e49f,
        0x5edef90e, 0x29d9c998, 0xb0d09822, 0xc7d7a8b4, 0x59b33d17, 0x2eb40d81,
        0xb7bd5c3b, 0xc0ba6cad, 0xedb88320, 0x9abfb3b6, 0x03b6e20c, 0x74b1d29a,
        0xead54739, 0x9dd277af, 0x04db2615, 0x73dc1683, 0xe3630b12, 0x94643b84,
        0x0d6d6a3e, 0x7a6a5aa8, 0xe40ecf0b, 0x9309ff9d, 0x0a00ae27, 0x7d079eb1,
        0xf00f9344, 0x8708a3d2, 0x1e01f268, 0x6906c2fe, 0xf762575d, 0x806567cb,
        0x196c3671, 0x6e6b06e7, 0xfed41b76, 0x89d32be0, 0x10da7a5a, 0x67dd4acc,
        0xf9b9df6f, 0x8ebeeff9, 0x17b7be43, 0x60b08ed5, 0xd6d6a3e8, 0xa1d1937e,
        0x38d8c2c4, 0x4fdff252, 0xd1bb67f1, 0xa6bc5767, 0x3fb506dd, 0x48b2364b,
        0xd80d2bda, 0xaf0a1b4c, 0x36034af6, 0x41047a60, 0xdf60efc3, 0xa867df55,
        0x316e8eef, 0x4669be79, 0xcb61b38c, 0xbc66831a, 0x256fd2a0, 0x5268e236,
        0xcc0c7795, 0xbb0b4703, 0x220216b9, 0x5505262f, 0xc5ba3bbe, 0xb2bd0b28,
        0x2bb45a92, 0x5cb36a04, 0xc2d7ffa7, 0xb5d0cf31, 0x2cd99e8b, 0x5bdeae1d,
        0x9b64c2b0, 0xec63f226, 0x756aa39c, 0x026d930a, 0x9c0906a9, 0xeb0e363f,
        0x72076785, 0x05005713, 0x95bf4a82, 0xe2b87a14, 0x7bb12bae, 0x0cb61b38,
        0x92d28e9b, 0xe5d5be0d, 0x7cdcefb7, 0x0bdbdf21, 0x86d3d2d4, 0xf1d4e242,
        0x68ddb3f8, 0x1fda836e, 0x81be16cd, 0xf6b9265b, 0x6fb077e1, 0x18b74777,
        0x88085ae6, 0xff0f6a70, 0x66063bca, 0x11010b5c, 0x8f659eff, 0xf862ae69,
        0x616bffd3, 0x166ccf45, 0xa00ae278, 0xd70dd2ee, 0x4e048354, 0x3903b3c2,
        0xa7672661, 0xd06016f7, 0x4969474d, 0x3e6e77db, 0xaed16a4a, 0xd9d65adc,
        0x40df0b66, 0x37d83bf0, 0xa9bcae53, 0xdebb9ec5, 0x47b2cf7f, 0x30b5ffe9,
        0xbdbdf21c, 0xcabac28a, 0x53b39330, 0x24b4a3a6, 0xbad03605, 0xcdd70693,
        0x54de5729, 0x23d967bf, 0xb3667a2e, 0xc4614ab8, 0x5d681b02, 0x2a6f2b94,
        0xb40bbe37, 0xc30c8ea1, 0x5a05df1b, 0x2d02ef8d
    };

static uint32_t CalculateCRC(uint8_t* data, uint32_t length)
{
	uint32_t crc = 0;
	uint32_t size = length;
	int index = 0;
	crc = crc ^ ~0U; //0xFFFFFFFF
	while (size-- > 0)
		crc = crc32_tab[(crc ^ data[index++]) & 0xFF] ^ (crc >> 8);

	return crc ^ ~0U; //0xFFFFFFFF
}
