/**
  ******************************************************************************
  * @file           : Variables.c
  * @brief          : Simple example for showing dmx functionality
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Kuenzler Technologies GbR
  * All rights reserved.
  *
  *
  ******************************************************************************
*/

/*==============================================================================*/
/* Includes                                                                  	*/
/*==============================================================================*/
#include "variables.h"
#include "string.h"
#include "stdio.h"
#include "managedflash.h"
#include "libIqMeshSpiInterface.h"

/*==============================================================================*/
/* Defines & Macros                                                           	*/
/*==============================================================================*/
#define CALLBACK_MAX 5

typedef struct
{
  notify_callback_t callbacks[CALLBACK_MAX];
  uint8_t registered_callbacks;
  uint8_t id;
} notify_t;

/*==============================================================================*/
/* Private functions                                                          	*/
/*==============================================================================*/

static bool CheckDataChanged(uint8_t* data, uint8_t* compareData, int length);
static void ClearVariable(void* data, uint8_t length, const void* defaultValue);
static void LoadVariable(uint8_t* index, void* data, uint8_t length, uint16_t key, uint8_t multiply,const void* defaultValue);

static void deregister_callback(notify_t *p_event_data, notify_callback_t callback);
static void register_callback(notify_t *p_event_data, notify_callback_t callback);
static void notify_callback(notify_t *p_event_data);

/*==============================================================================*/
/* Private variables                                                          	*/
/*==============================================================================*/


// Serial ---------------------------------
static uint64_t variableSerial;
#define KEY_VARIABLESSERIAL 0x0001
static uint8_t index_variableSerial = 0;
static notify_t callbacks_serial;
static bool flash_Serial;
static const uint64_t DEFAULTSERIALNUMBER = 1;


// DeviceLabel ---------------------------------
static device_label_t variableDeviceLabel;
#define KEY_VARIABLESDEVICELABEL 0x0002
static uint8_t index_variableDeviceLabel = 0;
static bool flash_DeviceLabel;

// Rdm Counter ---------------------------------
static rdm_counter_t variableCounter;
#define KEY_VARIABLESCOUNTER 0x0003
static uint8_t index_variableCounter = 0;
static bool flash_Counter;

// System State ---------------------------------
static system_state_t variableSystemState;
static notify_t callbacks_systemState;
static const system_state_t DEFAULT_SYSTEM_STATE_OFF = SYSTEM_STATE_OFF;

// Input Source ---------------------------------
static input_source_t variableInputSource;
#define KEY_VARIABLESINPUTSOURCE 0x0006
static uint8_t index_variableInputSource = 0;
static bool flash_InputSource;
static const uint8_t DEFAULTINPUTSOURCE = DMX_DEFAULT_INPUT_SOURCE;

// DMX Personality ---------------------------------
static dmx_settings_t variableDmxSettings;
#define KEY_VARIABLESDMXSETTINGS 0x0007
static uint8_t index_variableDmxSettings = 0;
static bool flash_DmxSettings;
static const dmx_settings_t DEFAULTDMXSETTINGS = {DMX_DEFAULT_PERSONALITY,DIMMER_CURVE_LINEAR, DMX_DEFAULT_STARTADDRESS,};

// Runtime Selection ---------------------------------
static features_t variableFeatures;
#define KEY_VARIABLESFEATURES 0x000A
static uint8_t index_variableFeatures = 0;
static bool flash_features;

// RDM Identify
static bool variableRdmIdentify;
static notify_t callbacks_rdmIdentify;

/*==============================================================================*/
/* Function definitions                                                     	*/
/*==============================================================================*/

void InitVariables()
{
	  // Clear Session Data
	  ClearVariable(&variableSystemState, sizeof(variableSystemState),&DEFAULT_SYSTEM_STATE_OFF);
	  ClearVariable(&variableRdmIdentify, sizeof(variableRdmIdentify),NULL);

	  InitFlash();
	  LoadVariable(&index_variableSerial, &variableSerial, sizeof(variableSerial), KEY_VARIABLESSERIAL, 10,&DEFAULTSERIALNUMBER);
	  LoadVariable(&index_variableDeviceLabel, &variableDeviceLabel, sizeof(variableDeviceLabel), KEY_VARIABLESDEVICELABEL, 10,RDM_DEFAULT_DEVICE_LABEL);
	  LoadVariable(&index_variableCounter, &variableCounter, sizeof(variableCounter), KEY_VARIABLESCOUNTER, 10, NULL);
	  LoadVariable(&index_variableInputSource, &variableInputSource, sizeof(variableInputSource), KEY_VARIABLESINPUTSOURCE, 10, &DEFAULTINPUTSOURCE);
	  LoadVariable(&index_variableDmxSettings, &variableDmxSettings, sizeof(variableDmxSettings), KEY_VARIABLESDMXSETTINGS, 10,&DEFAULTDMXSETTINGS);
	  LoadVariable(&index_variableFeatures, &variableFeatures, sizeof(variableFeatures), KEY_VARIABLESFEATURES, 10, NULL);
}

void VariablesTask()
{
	if(flash_Serial)
	{
		FlashWriteObject(index_variableSerial,(uint8_t*)&variableSerial,sizeof(variableSerial));
		flash_Serial = false;
		notify_callback(&callbacks_serial);
	}
	if(flash_DeviceLabel)
	{
		FlashWriteObject(index_variableDeviceLabel,(uint8_t*)&variableDeviceLabel,sizeof(variableDeviceLabel));
		flash_DeviceLabel = false;
	}
	if(flash_Counter)
	{
		FlashWriteObject(index_variableCounter,(uint8_t*)&variableCounter,sizeof(variableCounter));
		flash_Counter = false;
	}

	if(flash_InputSource)
	{
		FlashWriteObject(index_variableInputSource,(uint8_t*)&variableInputSource,sizeof(variableInputSource));
		flash_InputSource = false;
	}
	if(flash_DmxSettings)
	{
		FlashWriteObject(index_variableDmxSettings,(uint8_t*)&variableDmxSettings,sizeof(variableDmxSettings));
		flash_DmxSettings = false;
	}

	if(flash_features)
	{
		FlashWriteObject(index_variableFeatures,(uint8_t*)&variableFeatures,sizeof(variableFeatures));
		flash_features = false;
	}
//	if(flash_emergencyMode)
//	{
//		FlashWriteObject(index_variableEmergencyMode,(uint8_t*)&variableEmergencyMode,sizeof(variableEmergencyMode));
//		flash_emergencyMode = false;
//	}
//	if(flash_antitheftMode)
//	{
//		FlashWriteObject(index_variableAntitheftMode,(uint8_t*)&variableAntitheftMode,sizeof(variableAntitheftMode));
//		flash_antitheftMode = false;
//	}

}

//Serial Number
const uint64_t* get_serial(void)
{
	return (const uint64_t*)&variableSerial;
}
void set_serial(uint64_t serial)
{
	 variableSerial = serial;
	 bool changed = false;
	 if(!FlashObjectDiff(index_variableSerial,(uint8_t*)&variableSerial,sizeof(variableSerial),&changed))
	 {
		 //LOGGER_ERROR("Error while diffing file!");
		 changed = true;
	 }
	  if(changed)
	  {
		flash_Serial = true;
//		uint32_t err_code = app_sched_event_put(&callbacks_Serial, sizeof(callbacks_Serial), notify_callback_event_handler);
//		APP_ERROR_CHECK(err_code);
	  }
}

void register_notify_serial(notify_callback_t callback)
{
  register_callback(&callbacks_serial, callback);
}

void deregister_notify_serial(notify_callback_t callback)
{
	deregister_callback(&callbacks_serial, callback);
}


//RDMID
rdmid_t get_rdmid(void)
{
	rdmid_t id;
	//Calculate RDMID from Serial with our defined standard
	//  [8 bit - ESTA_Manufacturer_ID]
    //	[8 bit - ESTA_Manufacturer_ID]
    //  [3 bit - Interface|3 bit - Serial| 2 bit - Device ID]
    //	[8 bit - Device ID]
    //	[8 bit - Serial]
    //	[8 bit - Serial]
	id.array[0] = RDM_MANUFACTURER >> 8;
	id.array[1] = (uint8_t)RDM_MANUFACTURER;
	id.array[2] = (((variableSerial  >> 16) & 0x7) << 2) + (RDM_DEVICEMODEL_ID >> 8);
	id.array[3] = RDM_DEVICEMODEL_ID;
	id.array[4] = (uint8_t)(variableSerial  >> 8);
	id.array[5] = (uint8_t)(variableSerial);

	return id;
}

//Device Label
const device_label_t* get_deviceLabel(void)
{
	return (const device_label_t*)&variableDeviceLabel;
}
void set_deviceLabel(device_label_t label)
{
	variableDeviceLabel = label;
	bool changed = false;
	if(!FlashObjectDiff(index_variableDeviceLabel,(uint8_t*)&variableDeviceLabel,sizeof(variableDeviceLabel),&changed))
	{
	//LOGGER_ERROR("Error while diffing file!");
	changed = true;
	}
	if(changed)
	{
	flash_DeviceLabel = true;
	}
}

//Device Hours
const rdm_counter_t* get_counter(void)
{
	return (const rdm_counter_t*)&variableCounter;
}

void set_counter(rdm_counter_t counter)
{
	variableCounter = counter;
	bool changed = false;
	if(!FlashObjectDiff(index_variableCounter,(uint8_t*)&variableCounter,sizeof(variableCounter),&changed))
	{
	//LOGGER_ERROR("Error while diffing file!");
	changed = true;
	}
	if(changed)
	{
	flash_Counter = true;
	}
}

// System State ------------------------------------------------------

void set_systemState(system_state_t data)
{
	bool changed = CheckDataChanged((uint8_t*)&data, (uint8_t*)&variableSystemState,sizeof(variableSystemState));
	variableSystemState = data;
	if(changed)
	{
		notify_callback(&callbacks_systemState);
	}
}

const system_state_t* get_systemState()
{
	return (const system_state_t*)&variableSystemState;
}

void register_notify_systemState(notify_callback_t callback)
{
	register_callback(&callbacks_systemState, callback);
}

void deregister_notify_systemState(notify_callback_t callback)
{
	deregister_callback(&callbacks_systemState, callback);
}


// Input Source ------------------------------------------------------

void set_inputSource(input_source_t data)
{
  variableInputSource = data;
  bool changed = false;
  if(!FlashObjectDiff(index_variableInputSource,(uint8_t*)&variableInputSource,sizeof(variableInputSource),&changed))
  {
    //LOGGER_ERROR("Error while diffing file!");
    changed = true;
  }
  if(changed)
  {
    flash_InputSource = true;
  }
}

const input_source_t* get_inputSource()
{
  return (const input_source_t*)&variableInputSource;
}


// DMX Personality ------------------------------------------------------

bool set_dmxSettings(dmx_settings_t dmxData)
{
	if(dmxData.dmxPersonality > RDM_PERSONALITY_COUNT || dmxData.dmxPersonality == 0)
	{
		return false;
	}
	if(dmxData.dmxStartadress == 0 || dmxData.dmxStartadress > 512)
	{
		return false;
	}
	uint16_t footprintDMX = RDM_FOOTPRINTS[dmxData.dmxPersonality -1];//Get
	if(footprintDMX + dmxData.dmxStartadress > 512)
	{
		return false;
	}

	variableDmxSettings = dmxData;
	bool changed = false;
	if(!FlashObjectDiff(index_variableDmxSettings,(uint8_t*)&variableDmxSettings,sizeof(variableDmxSettings),&changed))
	{
	//LOGGER_ERROR("Error while diffing file!");
	changed = true;
	}
	if(changed)
	{
		flash_DmxSettings = true;
	}

	return true;
}

const dmx_settings_t* get_dmxSettings()
{
	if(variableDmxSettings.dmxPersonality == 0)
	{
		dmx_settings_t set = variableDmxSettings;
		set.dmxPersonality = 1;
		set_dmxSettings(set);
	}
  return (const dmx_settings_t*)&variableDmxSettings;
}


//Factory Defaults
bool get_factoryDefault()
{
	return !CheckDataChanged((uint8_t*)&variableDmxSettings, (uint8_t*)&DEFAULTDMXSETTINGS, sizeof(variableDmxSettings));
}

void set_factoryDefault()
{
	set_dmxSettings(DEFAULTDMXSETTINGS);
	iqMesh_unlinkIqData();
}

// Rdm Battery Runtime
const features_t* get_features()
{
	return (const features_t*)&variableFeatures;
}
void set_features(features_t data)
{
	variableFeatures = data;
	bool changed = false;
	if(!FlashObjectDiff(index_variableFeatures,(uint8_t*)&variableFeatures,sizeof(variableFeatures),&changed))
	{
		//LOGGER_ERROR("Error while diffing file!");
		changed = true;
	}

}

const bool* get_rdmIdentify(void)
{
	return (const bool*)&variableRdmIdentify;
}
void set_rdmIdentify(bool enable)
{
	bool changed = CheckDataChanged((uint8_t*)&enable, (uint8_t*)&variableRdmIdentify,sizeof(variableRdmIdentify));
	variableRdmIdentify = enable;
	if(changed)
	{
		notify_callback(&callbacks_rdmIdentify);
	}
}


static void notify_callback(notify_t *p_event_data)
{
  notify_t* callbacks = (notify_t*)p_event_data;
  uint8_t n = callbacks->registered_callbacks;
  for(uint8_t i = 0; i< CALLBACK_MAX;i++)
  {
    if(n == 0)
    {
      return;
    }
    if(callbacks->callbacks[i] != NULL)
    {
      n--;
      callbacks->callbacks[i]();
    }
  }
}

static void register_callback(notify_t *p_event_data, notify_callback_t callback)
{
	 for(uint8_t i = 0; i< CALLBACK_MAX;i++)
	  {
	    if(p_event_data->callbacks[i] == NULL)
	    {
	    	p_event_data->callbacks[i] = callback;
	    	p_event_data->registered_callbacks++;
	      return;
	    }
	  }
}

static void deregister_callback(notify_t *p_event_data, notify_callback_t callback)
{
	 if(p_event_data->registered_callbacks == 0)
	  {
	    //LOGGER_ERROR("No callback registered!");
	    return;
	  }
	  for(uint8_t i = 0; i< CALLBACK_MAX;i++)
	  {
	    if(p_event_data->callbacks[i] == callback)
	    {
	    	p_event_data->callbacks[i] = NULL;
	    	p_event_data->registered_callbacks--;
	      return;
	    }
	  }
	  //LOGGER_ERROR("Callback not registered");
}

static void ClearVariable(void* data, uint8_t length, const void* defaultValue)
{
	if(defaultValue != NULL)
	{
	  memcpy(data,defaultValue, length);
	}
	else
	{
	  memset(data,0,length);
	}
}

static void LoadVariable(uint8_t* index, void* data, uint8_t length, uint16_t key, uint8_t multiply, const void* defaultValue)
{
	  int objectIndex = FlashRegisterObject(key, length, multiply);
	  if(objectIndex < 0)
	  {
	    //LOGGER_ERROR("Object could not be registered!");
	    return;
	  }
	  *index = (uint8_t)objectIndex;

	  bool written = false;
	  bool success = FlashReadObject(*index, (uint8_t*)data, length, &written);
	  if(!success)
	  {
	    //LOGGER_ERROR("Object could not be read!");
	    return;
	  }

	  if(!written)
	  {
		  if(defaultValue != NULL)
		  {
			  memcpy(data,defaultValue, length);
		  }
		  else
		  {
			  memset(data,0,length);
		  }
	    //LOGGER_INFO("Session Fixture Information never written --> Default");
	  }
	  else
	  {
	    //LOGGER_INFO("Session Fixture Information loaded");
	  }
}

static bool CheckDataChanged(uint8_t* data, uint8_t* compareData, int length)
{
  for(int i= 0; i< length; i++)
  {
    if(data[i] != compareData[i])
    {
      return true;
    }
  }
  return false;
}
