/**
  ******************************************************************************
  * @file           : output_setting.c
  * @brief          : Simple example for output features like rdm identify, emergency light etc.
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Kuenzler Technologies GbR
  * All rights reserved.
  *
  *
  ******************************************************************************
*/

/*==============================================================================*/
/* Includes                                                                  	*/
/*==============================================================================*/
#include "output_setting.h"
#include "stdio.h"
#include "variables.h"
#include "stdlib.h"

/*==============================================================================*/
/* Defines & Macros                                                           	*/
/*==============================================================================*/
#define ALARMCOUNTER ((500 * 1000) /OUTPUT_REFRESH_RATE_US)
#define MINUTECOUNTER (MINUTES_MS /5)

/*==============================================================================*/
/* Private functions                                                          	*/
/*==============================================================================*/
static void OutputSettingCutPowerDraw(uint8_t* _red,uint8_t* _green,uint8_t* _blue);
static uint32_t CalculatePower(uint8_t* _red,uint8_t* _green,uint8_t* _blue);
/*==============================================================================*/
/* Private variables                                                          	*/
/*==============================================================================*/
static bool alarmColor;
static bool engineLight;
static uint32_t sourceCounter;
static bool timeSet;
static uint32_t lastTime;

/*==============================================================================*/
/* Function definitions                                                     	*/
/*==============================================================================*/
void InitOutputSetting()
{
	alarmColor = false;
	engineLight = false;
}

void OutputSettingTask(uint32_t tickMs)
{
	/* Count Source Minutes */
	if(!timeSet)
	{
		lastTime = tickMs;
		timeSet = true;
	}
	else if(tickMs >= lastTime + 5 || tickMs < lastTime) // count every 5 ms
	{
		if(engineLight)
		{
			sourceCounter++;
			lastTime = tickMs;
		}
	}
	if(sourceCounter >= MINUTECOUNTER)
	{
		rdm_counter_t counter = *get_counter();
		counter.sourceCounterMinutes++;
		set_counter(counter);
		sourceCounter = 0;
	}
	/* Toggle AlarmColor */
	alarmColor = ((tickMs%2000) < 1000);
}

void OutputSettingGetColorOverride(uint8_t* _red, uint8_t* _green, uint8_t* _blue, bool enabled)
{
	if(enabled)
	{
		if(*get_rdmIdentify() || (FEATURE_EMERGENCY_SUPPORT && (get_features()->emergencyMode && HardwareOutputSettingGpioReadPowerSupplyPin())))
		{
			for (int led= 0; led<OUTPUT_LED_COUNT; led++)
			{
				_red[led] = 0xFF;
				_green[led] = 0xFF;
				_blue[led] = 0xFF;
			}
			engineLight = true;
		}
		else
		{
			engineLight = false;
			for (int led= 0; led<OUTPUT_LED_COUNT; led++)
			{
				engineLight = (_red[led] > 0 || _green[led] > 0 || _blue[led] > 0);
				if(engineLight)
				{
					break;
				}
			}
		}
		OutputSettingCutPowerDraw(_red,_green,_blue);
	}
	else
	{
		engineLight = false;
	}
}

static void OutputSettingCutPowerDraw(uint8_t* _red,uint8_t* _green,uint8_t* _blue)
{
	/* Calculate Power draw and cut maximum output */
	if(FEATURE_RUNTIMESELECTION_SUPPORT && get_features()->runtimeSet != RDMIQBATTERYRUNTIME_FULLOUTPUT)
	{
		uint32_t allowedPowerDraw = ((uint32_t)BATTERY_CAPACITY_mWH / get_features()->runtimeSet) << 8;
		uint32_t powerdraw = CalculatePower(_red,_green,_blue);
		if(powerdraw > 0)
		{
			uint32_t faktor = allowedPowerDraw / powerdraw;
			if(faktor <= 256)
			{
				*_red = (uint8_t)((*_red * faktor) >> 8);
				*_green = (uint8_t)((*_green * faktor) >> 8);
				*_blue = (uint8_t)((*_blue * faktor) >> 8);
			}
		}
	}
}

static uint32_t CalculatePower(uint8_t* _red,uint8_t* _green,uint8_t* _blue)
{
	uint16_t avRed = 0;
	uint16_t avGreen = 0;
	uint16_t avBlue = 0;
	for (int led= 0; led<OUTPUT_LED_COUNT; led++)
	{
		avRed += _red[led];
		avGreen += _green[led];
		avBlue += _blue[led];
	}
	avRed /= OUTPUT_LED_COUNT;
	avGreen /= OUTPUT_LED_COUNT;
	avBlue /= OUTPUT_LED_COUNT;
	return (uint32_t)(((avRed * LED_POWER_RED_mW)/0xFFFF) + ((avGreen * LED_POWER_GREEN_mW)/0xFFFF) + ((avBlue * LED_POWER_BLUE_mW)/0xFFFF));
}


