#ifndef IQMESH_H__
#define IQMESH_H__

#include <stdbool.h>
#include <stdint.h>

typedef struct
{
	uint16_t uuid;
	uint16_t deviceCount;
	uint8_t signalQuality;
	uint8_t applicationId;
	char networkName[12];
	uint8_t networkIdentifier[3];
} iqMesh_information_t;

typedef enum {
	SPIEXT_IRQ_EDGE_RISING,
	SPIEXT_IRQ_EDGE_FALLING,
} iqmesh_irq_edge_t;

typedef enum {
	SPI_EVENT_COMPLETION,
	SPI_EVENT_ERROR,
} spi_event_t;

typedef enum { IQMESH_DFUFLAG_START, IQMESH_DFUFLAG_STOP } iqmesh_dfu_flag_t;

typedef enum {
	DFU_SUCCESS                             = 0x0,
	DFU_ERROR_FLASH_ERASE                   = 0x1,
	DFU_ERROR_NOT_IN_PROGRESS               = 0x2,
	DFU_ERROR_CORRUPTED_HEADER              = 0x3,
	DFU_ERROR_CORRUPTED_IMAGE               = 0x4,
	DFU_ERROR_INVALID_IMAGE_VERSION         = 0x5,
	DFU_ERROR_NO_PREAMBLE                   = 0x6,
	DFU_ERROR_INVALID_FIXTURE_ID            = 0x7,
	DFU_ERROR_UNKNOWN                       = 0x8,
	DFU_ERROR_SIGNATURE_NECESSARY_NOT_FOUND = 0x9,
	DFU_ERROR_NOT_IMPLEMENTED               = 0x0A,
	DFU_ERROR_WRONG_PACKETNR                = 0x0B
} iqmesh_dfu_response;

typedef enum {
	IQMESH_STATUS_UNLINKED,  // Unlinked
	IQMESH_STATUS_LINKED,    // linked
	IQMESH_STATUS_ACTIVE,    // Linked and active
	IQMESH_STATUS_MASTER,    // Linked and Master
	IQMESH_STATUS_INACTIVE,  // Linked and Inactive
} iqmesh_iq_status_t;

typedef enum {
	IQMESH_VISIBILITY_IF_SELECTED_INPUT_SOURCE,
	IQMESH_VISIBILITY_NON_ACTIVE_INPUT_SOURCE,
	IQMESH_VISIBILITY_ALWAYS,
} iqmesh_visibility_t;

typedef struct
{
	bool iqLinkedFlag : 1;
	bool iqRfLinkedFlag : 1;
	uint8_t reservedFlags : 6;
	iqmesh_iq_status_t iqStatus;
} iqmesh_status_flags_t;

typedef struct
{
	void (*DmxReceived)(uint8_t personality, const uint8_t *buf, uint16_t length);
	void (*RdmReceived)(const uint8_t *buf, uint16_t length);
	void (*DfuFlagReceived)(iqmesh_dfu_flag_t);
	void (*DfuDataReceived)(uint16_t packetNr, const uint8_t *buf, uint16_t length);
} iqmesh_callbacks_t;

bool iqMesh_init(const uint8_t *_rdm_id, uint16_t specialDmxLength, bool dfuSupport, iqmesh_callbacks_t callbacks);
bool iqMesh_task(uint32_t tickMs);
bool iqMesh_isBusy(uint32_t tickMs);
void iqMesh_extSpiEvent(spi_event_t event, uint8_t *buffer, uint16_t length);
void iqMesh_extIrqInterupt(iqmesh_irq_edge_t edge);
bool iqMesh_isModuleStarted(void);

void iqMesh_getInformation(iqMesh_information_t *info);
uint32_t iqMesh_getModuleBootloaderVersion(void);
uint32_t iqMesh_getModuleHardwareVersion(void);
uint32_t iqMesh_getModuleFirmwareVersion(void);
const uint8_t *iqMesh_getModuleCpuId(void);
iqmesh_status_flags_t iqMesh_getStatusFlags(void);

void iqMesh_unlinkIqData(void);
void iqMesh_setRdmId(const uint8_t *_rdm_id);
void iqMesh_startDfu_iqController(void);
void iqMesh_setVisibility(iqmesh_visibility_t visibility);
bool iqMesh_writeRdm(const uint8_t *msg, uint16_t length);
bool iqMesh_writeDmx(const uint8_t *_dmx, uint16_t length);
bool iqMesh_writeDfu(iqmesh_dfu_response dfuResponse);

/*==============================================================================*/
/* External defined hardware functions                                        	*/
/*==============================================================================*/
/* Start SPI Transmit and Receive interupt or DMA based */
extern void HardwareIqMeshSpiSetTxRx(uint8_t *buf, uint16_t length);
/* Set SPI CS Pin regarding set value */
extern void HardwareIqMeshSpiSetCsPin(bool set);
/* Read state of IRQ Gpio Pin */
extern bool HardwareIqMeshGpioReadIrqPin(void);

#endif
