/**
  ******************************************************************************
  * @file           : managedflash.h
  * @brief          : Simple example for showing flash functionality
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Kuenzler Technologies GbR
  * All rights reserved.
  *
  *
  ******************************************************************************
*/

/*==============================================================================*/
/* Includes                                                                  	*/
/*==============================================================================*/
#include "managedflash.h"
#include "string.h"
#include "stdlib.h"

/*==============================================================================*/
/* Defines & Macros                                                           	*/
/*==============================================================================*/
/* Make sure to define the necessary Defines in your application:              	*/
/* USER_FLASH_ADDRESS				e.g 0x08000000								*/
/* USER_FLASH_START					e.g 0x08067000								*/
/* USER_FLASH_SIZE																*/
/* USER_FLASH_PAGE_SIZE				e.g 0x1000 - 4KB							*/
/* USER_STORAGEOBJECTS				e.g 20										*/
/* USER_FLASH_BYTEALIGNMENT			e.g 8 (8Byte)								*/

#define STORAGEADDRESS		USER_STORAGE_ADDRESS
#define STORAGEADDRESSSTART USER_FLASH_START									
#define STORAGEADDRESSEND   (USER_FLASH_START + USER_FLASH_SIZE)
#define STORAGEPAGESIZE     USER_FLASH_PAGE_SIZE   								
#define STORAGESTARTPAGE	(USER_FLASH_START - USER_FLASH_ADDRESS) / STORAGEPAGESIZE
#define WRITTENFLAG         0xAA
#define FLASHPACKETSIZE     400													// Must be a multiple of 8
#define STORAGEPAGES      	(uint32_t)((STORAGEADDRESSEND - STORAGEADDRESSSTART) / STORAGEPAGESIZE)
#define STORAGEOBJECTS		USER_STORAGEOBJECTS									


#define FLASHSTORAGE_BYTEALIGNMENT	USER_FLASH_BYTEALIGNMENT					// e.g 8 Byte
#define OBJECTKEYWRITELENGTH        (((4 / FLASHSTORAGE_BYTEALIGNMENT) > 0)? 4:FLASHSTORAGE_BYTEALIGNMENT)

/*==============================================================================*/
/* Private functions                                                          	*/
/*==============================================================================*/
static int32_t FindObjectPlace(uint8_t objectId);
static bool FormatObject(uint8_t objectId);

/*==============================================================================*/
/* Private variables                                                          	*/
/*==============================================================================*/
static flash_object_t objects[STORAGEOBJECTS];
static uint8_t usedObjects = 0;

/*==============================================================================*/
/* Function definitions                                                     	*/
/*==============================================================================*/

void InitFlash()
{
  // Not used
}

int8_t FlashRegisterObject(uint16_t objectKey, uint16_t objectLength, uint32_t minWriteDivider)
{
  if(usedObjects < STORAGEOBJECTS)
  {
    // Get complete length with Flag and round up to sizeof(uint64_t)
    uint32_t completeLengthAligned = objectLength + 1;
    uint32_t m = completeLengthAligned % FLASHSTORAGE_BYTEALIGNMENT;
    if(m > 0)
    {
    	completeLengthAligned += FLASHSTORAGE_BYTEALIGNMENT - m;
    }
    // Calculate necessary Pages with object length * writeDiver + Keylength in alignment
    uint32_t wholeLength = (completeLengthAligned  * minWriteDivider + OBJECTKEYWRITELENGTH);
    uint32_t necessaryPages = wholeLength / STORAGEPAGESIZE;
    if(wholeLength % STORAGEPAGESIZE > 0)
      necessaryPages++;

    // Calculate maximum writes with these pages
    uint32_t usefulWriteDivider= ((necessaryPages * STORAGEPAGESIZE) - OBJECTKEYWRITELENGTH) / completeLengthAligned;

    // Get actualusedPages
    uint32_t usedPages = 0;
    if(usedObjects != 0)
    {
      usedPages = objects[usedObjects -1].startPage + objects[usedObjects -1].pageLength;
    }
    // Get available Pages
    uint32_t restPages = STORAGEPAGES - usedPages;

    // make sure there is enough space available
    if(necessaryPages <= restPages)
    {
        objects[usedObjects].startPage = (uint8_t)usedPages;
        objects[usedObjects].pageLength = (uint8_t)necessaryPages;
        objects[usedObjects].key = (uint32_t)((objectKey << 16) + objectLength);
        objects[usedObjects].length = objectLength;
        objects[usedObjects].completeLengthAligned = completeLengthAligned;
        objects[usedObjects].freePlaces = usefulWriteDivider;
        int32_t result = FindObjectPlace(usedObjects);
        if(result == -1)
        {
          // Object not found format it.
          FormatObject(usedObjects);
          objects[usedObjects].actualPlace = 0;
        }
        else
        {
          objects[usedObjects].actualPlace = (uint32_t)result;
        }

      //LOGGER_INFO("Register Object: %d , with Key: %d , Length: %d",usedObjects,objectKey,completeLengthInt);
      //LOGGER_INFO("                 Pages: %d, maxWrites: %d",necessaryPages,usefulWriteDivider);
        usedObjects++;
        return (int8_t)(usedObjects -1);
    }
  //LOGGER_ERROR("Error: No space available");
    return -1;
  }
//LOGGER_ERROR("Error: No object available");
  return -1;
}

bool FlashReadObject(uint8_t objectId, uint8_t* data, uint16_t objectLength, bool* written)
{
  if(objectId < usedObjects)
  {
    if(objects[objectId].length == objectLength)
    {
       if(objects[objectId].actualPlace > 0)
       {
             uint32_t pageAddress = STORAGEADDRESSSTART + objects[objectId].startPage * STORAGEPAGESIZE;
             uint32_t placeAddress = pageAddress + OBJECTKEYWRITELENGTH + (objects[objectId].actualPlace -1) * objects[objectId].completeLengthAligned;
             uint8_t readBuffer[objects[objectId].completeLengthAligned];
             memcpy(readBuffer,(uint8_t*)placeAddress,objects[objectId].completeLengthAligned);
             if(readBuffer[0] == WRITTENFLAG)
             {
                memcpy(data,&readBuffer[1], objectLength);
                *written = true;
                //LOGGER_INFO("Successful read Object: %d at place: %d",objectId, objects[objectId].actualPlace -1);
                //LOGGER_DUMP("Data:",data, objectLength);
                return true;
             }

        //LOGGER_ERROR("Error: Object format error!");
          return false;
       }
     //LOGGER_INFO("Object not written!");
       *written = false;
       return true;
    }
  //LOGGER_ERROR("Error: Object length does not match!");
    return false;
  }
//LOGGER_ERROR("Error: Object not registered!");
  return false;
}

bool FlashGetObjectPointer(uint8_t objectId, uint8_t** data, bool* written)
{
  if(objectId < usedObjects)
  {
       if(objects[objectId].actualPlace > 0)
       {
             uint32_t pageAddress = STORAGEADDRESSSTART + objects[objectId].startPage * STORAGEPAGESIZE;
             uint32_t placeAddress = pageAddress + OBJECTKEYWRITELENGTH + (objects[objectId].actualPlace -1) * objects[objectId].completeLengthAligned;
             uint8_t readBuffer[4];
             memcpy(readBuffer,(uint8_t*)placeAddress,4);
             if(readBuffer[0] == WRITTENFLAG)
             {
                *data = (uint8_t*)(placeAddress + 1);
                *written = true;
              //LOGGER_INFO("Successful Get Pointer Object: %d at place: %d",objectId, objects[objectId].actualPlace -1);
                return true;
             }

        //LOGGER_ERROR("Error: Object format error!");
          return false;
       }
     //LOGGER_INFO("Object not written!");
       *written = false;
       return true;
  }
//LOGGER_ERROR("Error: Object not registered!");
  return false;
}

bool FlashObjectDiff(uint8_t objectId, uint8_t* data, uint16_t bufferLength, bool *changed)
{
  if(objectId < usedObjects)
  {
    if(bufferLength == objects[objectId].length)
    {
      bool written = false;
      uint8_t buffer[bufferLength];
      bool success = FlashReadObject(objectId, buffer, bufferLength, &written);
      if(success)
      {
        if(written)
        {
          for(uint8_t i=0;i< bufferLength;i++)
          {
            if(data[i] != buffer[i])
            {
              *changed = true;
            }
          }
        }
        else
        {
           *changed = true;
        }
        return true;
      }
      return false;
    }
  }
  return false;
}

bool FlashWriteObject(uint8_t objectId, uint8_t* data, uint16_t bufferLength)
{
  if(objectId < usedObjects)
  {
    if(bufferLength == objects[objectId].length)
    {
      if(objects[objectId].actualPlace >= objects[objectId].freePlaces)
      {
        FormatObject(objectId);
        objects[objectId].actualPlace = 0;
      }
      uint32_t pageAddress = STORAGEADDRESSSTART + objects[objectId].startPage * STORAGEPAGESIZE;
      uint32_t placeAddress = pageAddress + OBJECTKEYWRITELENGTH + (objects[objectId].actualPlace) * objects[objectId].completeLengthAligned;

      uint32_t packets = (uint32_t)((bufferLength + 1) / FLASHPACKETSIZE);
      uint32_t rest = (bufferLength + 1) - (packets * FLASHPACKETSIZE);
      uint16_t lengthAligned = 0;
      if(packets == 0)
      {
    	lengthAligned = (uint16_t)(objects[objectId].completeLengthAligned / FLASHSTORAGE_BYTEALIGNMENT);
    	uint8_t* writeBuffer = malloc(lengthAligned * FLASHSTORAGE_BYTEALIGNMENT);
        writeBuffer[0] = WRITTENFLAG;
        memcpy(&writeBuffer[1], data, bufferLength);

        // Flash Write ----------------------------------------------------------
		if(!HardwareWriteFlash(placeAddress,lengthAligned,writeBuffer))
		{
			free(writeBuffer);
			return false;
		}
		free(writeBuffer);
      }
      else
      {
		lengthAligned = FLASHPACKETSIZE / FLASHSTORAGE_BYTEALIGNMENT;
		uint8_t* writeBuffer = malloc(lengthAligned * FLASHSTORAGE_BYTEALIGNMENT);
		writeBuffer[0] = WRITTENFLAG;
		memcpy(&(writeBuffer[1]), data, FLASHPACKETSIZE -1);
		// Flash Write ----------------------------------------------------------
		if(!HardwareWriteFlash(placeAddress,lengthAligned,writeBuffer))
		{
			free(writeBuffer);
			return false;
		}
		
		for(uint8_t i= 0; i< packets - 1;i++)
		{
			memcpy(writeBuffer, &data[FLASHPACKETSIZE -1 + i * FLASHPACKETSIZE], FLASHPACKETSIZE);
			if(!HardwareWriteFlash((uint32_t)(placeAddress + (i +1) * FLASHPACKETSIZE),lengthAligned,writeBuffer))
			{
				free(writeBuffer);
				return false;
			}
		}

		// Last Packet
		memset(writeBuffer,0,FLASHPACKETSIZE);
		memcpy(writeBuffer, &data[FLASHPACKETSIZE -1 + (packets - 1) * FLASHPACKETSIZE], rest);

		uint32_t writeRestaligned = (objects[objectId].completeLengthAligned - (bufferLength + 1) + rest) / FLASHSTORAGE_BYTEALIGNMENT;
		
		if(!HardwareWriteFlash((placeAddress + (packets * FLASHPACKETSIZE)),writeRestaligned,writeBuffer))
		{
			free(writeBuffer);
			return false;
		}
		free(writeBuffer);
      }
    //LOGGER_INFO("Successful written Object: %d at place: %d",objectId, objects[objectId].actualPlace);
      // Set actual Place up
      objects[objectId].actualPlace++;
      return true;
    }
  //LOGGER_ERROR("Error: Object length does not match!");
    return false;
  }
//LOGGER_ERROR("Error: Object not registered!");
  return false;
}

int32_t FlashGetHistoryCount(uint8_t objectId)
{
  // Read object Key
	if(objectId < usedObjects)
	{
		return objects[objectId].actualPlace;
	}
	else
	{
	//LOGGER_ERROR("ObjectKey not found!");
		return -1;
	}
}

bool FlashReadObjectHistory(uint8_t objectId, uint8_t* data, uint16_t objectLength, uint32_t historyIndex)
{
  if(objectId < usedObjects)
  {
    if(objects[objectId].length == objectLength)
    {
       if(objects[objectId].actualPlace > 0 && objects[objectId].actualPlace > historyIndex)
       {
             uint32_t pageAddress = STORAGEADDRESSSTART + objects[objectId].startPage * STORAGEPAGESIZE;
             uint32_t placeAddress = pageAddress + OBJECTKEYWRITELENGTH + (historyIndex) * objects[objectId].completeLengthAligned;
             uint8_t readBuffer[objects[objectId].completeLengthAligned];
             memcpy(readBuffer,(uint8_t*)placeAddress,objects[objectId].completeLengthAligned);
             if(readBuffer[0] == WRITTENFLAG)
             {
                memcpy(data,&readBuffer[1], objectLength);
                //LOGGER_INFO("Successful read Object: %d at place: %d",objectId, objects[objectId].actualPlace -1);
                //LOGGER_DUMP("Data:",data, objectLength);
                return true;
             }
        //LOGGER_ERROR("Error: Object format error!");
          return false;
       }
     //LOGGER_INFO("Object not written!");
       return false;
    }
  //LOGGER_ERROR("Error: Object length does not match!");
    return false;
  }
//LOGGER_ERROR("Error: Object not registered!");
  return false;
}

// ---- Internal ------------------------------------------

static int32_t FindObjectPlace(uint8_t objectId)
{
  // Read object Key
  uint32_t objectKey = 0;
  uint32_t pageAddress = STORAGEADDRESSSTART + objects[objectId].startPage * STORAGEPAGESIZE;

  memcpy(&objectKey,(uint8_t*)pageAddress,4);

  if(objectKey == objects[objectId].key)
  {
    for(uint32_t i= 0; i< objects[objectId].freePlaces;i++)
    {
      uint32_t placeAddress = pageAddress + OBJECTKEYWRITELENGTH + i * objects[objectId].completeLengthAligned;
      uint8_t flag = 0;
      memcpy(&flag,(uint8_t*)placeAddress,1);
      if(flag == 0xFF)
      {
      //LOGGER_INFO("Object: %d found at place: %d",objectId, i);
        return (int32_t)i;
      }
      if(flag != WRITTENFLAG)
      {
        return -1;
      //LOGGER_ERROR("Space is not formatted correcty!");
      }
    }
    return (int32_t)objects[objectId].freePlaces;
  }

//LOGGER_ERROR("ObjectKey not found!");
  return -1;
}

static bool FormatObject(uint8_t objectId)
{
	uint32_t pageAddress = STORAGEADDRESSSTART + objects[objectId].startPage * STORAGEPAGESIZE;
	uint16_t pageIndex = STORAGESTARTPAGE + objects[objectId].startPage;
	
	if(!HardwareEraseFlash(pageAddress,pageIndex, objects[objectId].pageLength))
	{
		return false;
	}
	
	uint64_t writeKey = 0;
	memcpy(&writeKey,&objects[objectId].key,sizeof(objects[objectId].key));
	uint16_t AlignedLength = (uint16_t)(OBJECTKEYWRITELENGTH / FLASHSTORAGE_BYTEALIGNMENT);
	
	if(!HardwareWriteFlash(pageAddress,AlignedLength,&writeKey))
	{
		return false;
	}
	return true;
	 //LOGGER_INFO("Object: %d formatted",objectId);
}

/*  Example Hardware Routine
bool HardwareWriteFlash(uint32_t placeAddress, uint16_t lengthAligned, void* writeBuffer)
{
	HAL_StatusTypeDef ret;
	if( HAL_FLASH_Unlock() != HAL_OK )
	{
		return false;
	}
	ret = FLASH_WaitForLastOperation( HAL_MAX_DELAY );

	for( uint32_t i = 0u; i < lengthAligned; i++)
	{
		ret = HAL_FLASH_Program(FLASH_TYPEPROGRAM_DOUBLEWORD,
						   placeAddress + (i * FLASHSTORAGE_BYTEALIGNMENT),
						   *((uint64_t*)(&((uint8_t*)writeBuffer)[i * FLASHSTORAGE_BYTEALIGNMENT])));
		if( ret != HAL_OK )
		{
		 return false;
		}
	}

	ret = FLASH_WaitForLastOperation( HAL_MAX_DELAY );

	if( ret != HAL_OK )
	{
		return false;
	}

	ret = HAL_FLASH_Lock();
	if( ret != HAL_OK )
	{
		return false;
	}
	return true;
}

bool HardwareEraseFlash(uint32_t flashPageAddress, uint32_t flashStartPage, uint32_t flashPages)
{
	HAL_StatusTypeDef ret;
	if( HAL_FLASH_Unlock() != HAL_OK )
	{
		return false;
	}
	ret = FLASH_WaitForLastOperation(HAL_MAX_DELAY);

	//Erase the Flash
	FLASH_EraseInitTypeDef EraseInitStruct;
	uint32_t SectorError;

	EraseInitStruct.TypeErase   = FLASH_TYPEERASE_PAGES;
	EraseInitStruct.Banks       = FLASH_BANK_1;
	EraseInitStruct.Page		= flashStartPage;
	// Pages needed for Application + one Page for saving the settings
	EraseInitStruct.NbPages     = flashPages;
	// clear all flags before you write it to flash
	//__HAL_FLASH_CLEAR_FLAG(FLASH_FLAG_EOP | FLASH_FLAG_OPERR |
	//			FLASH_FLAG_WRPERR | FLASH_FLAG_PGAERR | FLASH_FLAG_PGPERR);

	if(HAL_FLASHEx_Erase( &EraseInitStruct, &SectorError )!= HAL_OK )
	{
		return false;
	}

	//Check if the FLASH_FLAG_BSY.
	ret = FLASH_WaitForLastOperation( HAL_MAX_DELAY );

	if( ret != HAL_OK )
	{
		return false;
	}
	ret = HAL_FLASH_Lock();
	if( ret != HAL_OK )
	{
		return false;
	}
	return true;
}

*/
