/**
  ******************************************************************************
  * @file           : Variables.c
  * @brief          : Simple example for showing dmx functionality
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Kuenzler Technologies GbR
  * All rights reserved.
  *
  *
  ******************************************************************************
*/

/*==============================================================================*/
/* Includes                                                                  	*/
/*==============================================================================*/
#include "variables.h"
#include "string.h"
#include "stdio.h"
#include "managedflash.h"

/*==============================================================================*/
/* Defines & Macros                                                           	*/
/*==============================================================================*/
#define CALLBACK_MAX 5

typedef struct
{
  notify_callback_t callbacks[CALLBACK_MAX];
  uint8_t registered_callbacks;
  uint8_t id;
} notify_t;

/*==============================================================================*/
/* Private functions                                                          	*/
/*==============================================================================*/

static bool CheckDataChanged(uint8_t* data, uint8_t* compareData, int length);
static void ClearVariable(void* data, uint8_t length, const void* defaultValue);
static void LoadVariable(uint8_t* index, void* data, uint8_t length, uint16_t key, uint8_t multiply,const void* defaultValue);

static void deregister_callback(notify_t *p_event_data, notify_callback_t callback);
static void register_callback(notify_t *p_event_data, notify_callback_t callback);
static void notify_callback(notify_t *p_event_data);

/*==============================================================================*/
/* Private variables                                                          	*/
/*==============================================================================*/


// Serial ---------------------------------
static uint64_t variableSerial;
#define KEY_VARIABLESSERIAL 0x0001
static uint8_t index_variableSerial = 0;
static notify_t callbacks_serial;
static bool flash_Serial;
static const uint64_t DEFAULTSERIALNUMBER = 1;

// DeviceLabel ---------------------------------
static device_label_t variableDeviceLabel;
#define KEY_VARIABLESDEVICELABEL 0x0002
static uint8_t index_variableDeviceLabel = 0;
static bool flash_DeviceLabel;

// Device Minutes ---------------------------------
static uint32_t variableDeviceMinutes;
#define KEY_VARIABLESDEVCICEHOURS 0x0003
static uint8_t index_variableDeviceMinutes = 0;
static bool flash_DeviceMinutes;

// Source Minutes ---------------------------------
static uint32_t variableSourceMinutes;
#define KEY_VARIABLESSOURCEHOURS 0x0004
static uint8_t index_variableSourceMinutes = 0;
static bool flash_SourceMinutes;

// Powercycles ---------------------------------
static uint32_t variablePowerCycles;
#define KEY_VARIABLESPOWERCYCLES 0x0005
static uint8_t index_variablePowerCycles = 0;
static bool flash_PowerCycles;

// DMX Personality ---------------------------------
static uint8_t variableDmxPersonality;
#define KEY_VARIABLESDMXPERSONALITY 0x0007
static uint8_t index_variableDmxPersonality = 0;
static bool flash_DmxPersonality;
static const uint8_t DEFAULTPERSONALITY = DMX_DEFAULT_PERSONALITY;

// DMX Startadress ---------------------------------
static uint16_t variableDmxStartadress;
#define KEY_VARIABLESDMXSTARTADRESS 0x0008
static uint8_t index_variableDmxStartadress = 0;
static bool flash_DmxStartadress;
static const uint16_t DEFAULTSTARTADDRESS = DMX_DEFAULT_STARTADDRESS;

// NTC Beta ---------------------------------
static float variableNtcBeta;
#define KEY_VARIABLESNTCBETA 0x000A
static uint8_t index_variableNtcBeta = 0;
static bool flash_NtcBeta;
static const float DEFAULTNTCBETA = 3800;

// Driver Config ---------------------------------
static cc_key_store_t variableCCKey;
#define KEY_VARIABLESCCKEY 0x000B
static uint8_t index_variableCCKey = 0;
static bool flash_CCKey;


static cc_calibration_store_t variableCCCal;
#define KEY_VARIABLESCCCAL 0x000C
static uint8_t index_variableCCCal = 0;
static bool flash_CCCal;

// RDM Identify ---------------------------------
static bool variableRdmIdentify;
static notify_t callbacks_rdmIdentify;
static bool changed_rdmIdentify;
static const uint16_t DEFAULTCAL[LED_COUNT][DIE_COUNT] = {{0x7FFF,0x7FFF,0x7FFF,0x7FFF,0x7FFF}};
/*==============================================================================*/
/* Function definitions                                                     	*/
/*==============================================================================*/

void InitVariables()
{
	  // Clear Session Data
	  ClearVariable(&variableRdmIdentify, sizeof(variableRdmIdentify),NULL);

	  InitFlash();
	  LoadVariable(&index_variableSerial, &variableSerial, sizeof(variableSerial), KEY_VARIABLESSERIAL, 10,&DEFAULTSERIALNUMBER);
	  LoadVariable(&index_variableDeviceLabel, &variableDeviceLabel, sizeof(variableDeviceLabel), KEY_VARIABLESDEVICELABEL, 10,RDM_DEFAULT_DEVICE_LABEL);
	  LoadVariable(&index_variableDmxPersonality, &variableDmxPersonality, sizeof(variableDmxPersonality), KEY_VARIABLESDMXPERSONALITY,10, &DEFAULTPERSONALITY);
	  LoadVariable(&index_variableDmxStartadress, &variableDmxStartadress,sizeof(variableDmxStartadress), KEY_VARIABLESDMXSTARTADRESS,10, &DEFAULTSTARTADDRESS);
	  LoadVariable(&index_variableDeviceMinutes, &variableDeviceMinutes, sizeof(variableDeviceMinutes), KEY_VARIABLESDEVCICEHOURS, 10, NULL);
	  LoadVariable(&index_variableSourceMinutes, &variableSourceMinutes, sizeof(variableSourceMinutes), KEY_VARIABLESSOURCEHOURS, 10, NULL);
	  LoadVariable(&index_variablePowerCycles, &variablePowerCycles, sizeof(variablePowerCycles), KEY_VARIABLESPOWERCYCLES, 10, NULL);
	  LoadVariable(&index_variableNtcBeta, &variableNtcBeta, sizeof(variableNtcBeta), KEY_VARIABLESNTCBETA, 10, &DEFAULTNTCBETA);
	  LoadVariable(&index_variableCCKey, &variableCCKey, sizeof(variableCCKey), KEY_VARIABLESCCKEY, 10, NULL);
	  LoadVariable(&index_variableCCCal, &variableCCCal, sizeof(variableCCCal), KEY_VARIABLESCCCAL, 10, &DEFAULTCAL);
}

void VariablesTask()
{
	if(flash_Serial)
	{
		FlashWriteObject(index_variableSerial,(uint8_t*)&variableSerial,sizeof(variableSerial));
		flash_Serial = false;
	}
	if(flash_DeviceLabel)
	{
		FlashWriteObject(index_variableDeviceLabel,(uint8_t*)&variableDeviceLabel,sizeof(variableDeviceLabel));
		flash_DeviceLabel = false;
	}
	if(flash_DeviceMinutes)
	{
		FlashWriteObject(index_variableDeviceMinutes,(uint8_t*)&variableDeviceMinutes,sizeof(variableDeviceMinutes));
		flash_DeviceMinutes = false;
	}
	if(flash_SourceMinutes)
	{
		FlashWriteObject(index_variableSourceMinutes,(uint8_t*)&variableSourceMinutes,sizeof(variableSourceMinutes));
		flash_SourceMinutes = false;
	}
	if(flash_PowerCycles)
	{
		FlashWriteObject(index_variablePowerCycles,(uint8_t*)&variablePowerCycles,sizeof(variablePowerCycles));
		flash_PowerCycles = false;
	}
	if(flash_DmxPersonality)
	{
		FlashWriteObject(index_variableDmxPersonality,(uint8_t*)&variableDmxPersonality,sizeof(variableDmxPersonality));
		flash_DmxPersonality = false;
	}
	if(flash_DmxStartadress)
	{
		FlashWriteObject(index_variableDmxStartadress,(uint8_t*)&variableDmxStartadress,sizeof(variableDmxStartadress));
		flash_DmxStartadress = false;
	}
	if(flash_NtcBeta)
	{
		FlashWriteObject(index_variableNtcBeta,(uint8_t*)&variableNtcBeta,sizeof(variableNtcBeta));
		flash_NtcBeta = false;
	}
	if(flash_CCKey)
	{
		FlashWriteObject(index_variableCCKey,(uint8_t*)&variableCCKey,sizeof(variableCCKey));
		flash_CCKey = false;
	}
	if(flash_CCCal)
	{
		FlashWriteObject(index_variableCCCal,(uint8_t*)&variableCCCal,sizeof(variableCCCal));
		flash_CCCal = false;
	}
	if(changed_rdmIdentify)
	{
		notify_callback(&callbacks_rdmIdentify);
		changed_rdmIdentify = false;
	}
}

//Serial Number
const uint64_t* get_serial(void)
{
	return (const uint64_t*)&variableSerial;
}
void set_serial(uint64_t serial)
{
	 variableSerial = serial;
	 bool changed = false;
	 if(!FlashObjectDiff(index_variableSerial,(uint8_t*)&variableSerial,sizeof(variableSerial),&changed))
	 {
		 //LOGGER_ERROR("Error while diffing file!");
		 changed = true;
	 }
	  if(changed)
	  {
		flash_Serial = true;
	  }
}

void register_notify_serial(notify_callback_t callback)
{
  register_callback(&callbacks_serial, callback);
}

void deregister_notify_serial(notify_callback_t callback)
{
	deregister_callback(&callbacks_serial, callback);
}


//RDMID
rdmid_t get_rdmid(void)
{
	rdmid_t id;
	//Calculate RDMID from Serial with our defined standard
	//  [8 bit - ESTA_Manufacturer_ID]
    //	[8 bit - ESTA_Manufacturer_ID]
    //  [3 bit - Interface|3 bit - Serial| 2 bit - Device ID]
    //	[8 bit - Device ID]
    //	[8 bit - Serial]
    //	[8 bit - Serial]
	id.array[0] = RDM_MANUFACTURER >> 8;
	id.array[1] = (uint8_t)RDM_MANUFACTURER;
	id.array[2] = (((variableSerial  >> 16) & 0x7) << 2) + (RDM_DEVICEMODEL_ID >> 8);
	id.array[3] = RDM_DEVICEMODEL_ID;
	id.array[4] = (uint8_t)(variableSerial  >> 8);
	id.array[5] = (uint8_t)(variableSerial);

	return id;
}

//Device Label
const device_label_t* get_deviceLabel(void)
{
	return (const device_label_t*)&variableDeviceLabel;
}
void set_deviceLabel(device_label_t label)
{
	variableDeviceLabel = label;
	bool changed = false;
	if(!FlashObjectDiff(index_variableDeviceLabel,(uint8_t*)&variableDeviceLabel,sizeof(variableDeviceLabel),&changed))
	{
		//LOGGER_ERROR("Error while diffing file!");
		changed = true;
	}
	if(changed)
	{
		flash_DeviceLabel = true;
	}
}

//Device Hours
const uint32_t* get_deviceMinutes(void)
{
	return (const uint32_t*)&variableDeviceMinutes;
}

void set_deviceMinutes(uint32_t hours)
{
	variableDeviceMinutes = hours;
	bool changed = false;
	if(!FlashObjectDiff(index_variableDeviceMinutes,(uint8_t*)&variableDeviceMinutes,sizeof(variableDeviceMinutes),&changed))
	{
		//LOGGER_ERROR("Error while diffing file!");
		changed = true;
	}
	if(changed)
	{
		flash_DeviceMinutes = true;
	}
}

//Lamp / Source Hours
const uint32_t* get_sourceMinutes(void)
{
	return (const uint32_t*)&variableSourceMinutes;
}
void set_sourceMinutes(uint32_t hours)
{
	variableSourceMinutes = hours;
	bool changed = false;
	if(!FlashObjectDiff(index_variableSourceMinutes,(uint8_t*)&variableSourceMinutes,sizeof(variableSourceMinutes),&changed))
	{
		//LOGGER_ERROR("Error while diffing file!");
		changed = true;
	}
	if(changed)
	{
		flash_SourceMinutes = true;
	}
}

//Power Cycles
const uint32_t* get_powerCycles(void)
{
	return (const uint32_t*)&variablePowerCycles;
}
void set_powerCycles(uint32_t cycles)
{
  variablePowerCycles = cycles;
  bool changed = false;
  if(!FlashObjectDiff(index_variablePowerCycles,(uint8_t*)&variablePowerCycles,sizeof(variablePowerCycles),&changed))
  {
	//LOGGER_ERROR("Error while diffing file!");
	changed = true;
  }
  if(changed)
  {
	flash_PowerCycles = true;
  }
}

// DMX Personality ------------------------------------------------------

bool set_dmxPersonality(uint8_t data)
{
	if(data >= RDM_PERSONALITY_COUNT + 1 || data == 0)
	{
		return false;
	}
	variableDmxPersonality = data;
	bool changed = false;
	if(!FlashObjectDiff(index_variableDmxPersonality,(uint8_t*)&variableDmxPersonality,sizeof(variableDmxPersonality),&changed))
	{
		//LOGGER_ERROR("Error while diffing file!");
		changed = true;
	}
	if(changed)
	{
		flash_DmxPersonality = true;
	}
	// Make sure dmxaddress is valid
	uint16_t footprintDMX = RDM_FOOTPRINTS[data -1];//Get
	if(footprintDMX + *get_dmxStartadress() > 512)
	{
		set_dmxStartadress(512 - footprintDMX);
	}
	return true;
}

const uint8_t* get_dmxPersonality()
{
	if(variableDmxPersonality == 0)
	{
		set_dmxPersonality(1);
	}
  return (const uint8_t*)&variableDmxPersonality;
}

// DMX Personality ------------------------------------------------------

void set_dmxStartadress(uint16_t data)
{
	uint16_t footprintDMX = RDM_FOOTPRINTS[variableDmxPersonality -1];//Get
	if(footprintDMX + *get_dmxStartadress() > 512)
	{
		data = (512 - footprintDMX);
	}
	variableDmxStartadress = data;
	bool changed = false;
	if(!FlashObjectDiff(index_variableDmxStartadress,(uint8_t*)&variableDmxStartadress,sizeof(variableDmxStartadress),&changed))
	{
		//LOGGER_ERROR("Error while diffing file!");
		changed = true;
	}
	if(changed)
	{
		flash_DmxStartadress = true;
	}
}

const uint16_t* get_dmxStartadress()
{
  return (const uint16_t*)&variableDmxStartadress;
}


// NTC Beta ------------------------------------------------------

void set_ntcBeta(float data)
{
  variableNtcBeta = data;
  bool changed = false;
  if(!FlashObjectDiff(index_variableNtcBeta,(uint8_t*)&variableNtcBeta,sizeof(variableNtcBeta),&changed))
  {
    //LOGGER_ERROR("Error while diffing file!");
    changed = true;
  }
  if(changed)
  {
    flash_NtcBeta = true;
  }
}


const float* get_ntcBeta()
{
  return (const float*)&variableNtcBeta;
}

//Factory Defaults ---------------------------------

bool get_factoryDefault()
{
	if(variableDmxPersonality != DMX_DEFAULT_PERSONALITY)
	{
		return false;
	}
	if(variableDmxStartadress != DMX_DEFAULT_STARTADDRESS)
	{
		return false;
	}
	return true;
}

void set_factoryDefault()
{
	set_dmxPersonality(DMX_DEFAULT_PERSONALITY);
	set_dmxStartadress(DMX_DEFAULT_STARTADDRESS);
}

// CC Key ------------------------------------------------------

void set_ccKey(cc_key_store_t data)
{
  variableCCKey = data;
  bool changed = false;
  if(!FlashObjectDiff(index_variableCCKey,(uint8_t*)&variableCCKey,sizeof(variableCCKey),&changed))
  {
    //LOGGER_ERROR("Error while diffing file!");
    changed = true;
  }
  if(changed)
  {
    flash_CCKey = true;
  }
}


const cc_key_store_t* get_ccKey()
{
  return (const cc_key_store_t*)&variableCCKey;
}

// CC Key ------------------------------------------------------

void set_ccCal(cc_calibration_store_t data)
{
  variableCCCal = data;
  bool changed = false;
  if(!FlashObjectDiff(index_variableCCCal,(uint8_t*)&variableCCCal,sizeof(variableCCCal),&changed))
  {
    //LOGGER_ERROR("Error while diffing file!");
    changed = true;
  }
  if(changed)
  {
    flash_CCCal = true;
  }
}


const cc_calibration_store_t* get_ccCal()
{
  return (const cc_calibration_store_t*)&variableCCCal;
}

// Get Rdm Identify -----------------------------------

const bool* get_rdmIdentify(void)
{
	return (const bool*)&variableRdmIdentify;
}
void set_rdmIdentify(bool enable)
{
	bool changed = CheckDataChanged((uint8_t*)&enable, (uint8_t*)&variableRdmIdentify,sizeof(variableRdmIdentify));
	variableRdmIdentify = enable;
	if(changed)
	{
		changed_rdmIdentify = true;
	}
}

void register_notify_rdmIdentify(notify_callback_t callback)
{
	register_callback(&callbacks_rdmIdentify, callback);
}

void deregister_notify_rdmIdentify(notify_callback_t callback)
{
	deregister_callback(&callbacks_rdmIdentify, callback);
}


static void notify_callback(notify_t *p_event_data)
{
  notify_t* callbacks = (notify_t*)p_event_data;
  uint8_t n = callbacks->registered_callbacks;
  for(uint8_t i = 0; i< CALLBACK_MAX;i++)
  {
    if(n == 0)
    {
      return;
    }
    if(callbacks->callbacks[i] != NULL)
    {
      n--;
      callbacks->callbacks[i]();
    }
  }
}

static void register_callback(notify_t *p_event_data, notify_callback_t callback)
{
	 for(uint8_t i = 0; i< CALLBACK_MAX;i++)
	  {
	    if(p_event_data->callbacks[i] == NULL)
	    {
	    	p_event_data->callbacks[i] = callback;
	    	p_event_data->registered_callbacks++;
	      return;
	    }
	  }
}

static void deregister_callback(notify_t *p_event_data, notify_callback_t callback)
{
	 if(p_event_data->registered_callbacks == 0)
	  {
	    //LOGGER_ERROR("No callback registered!");
	    return;
	  }
	  for(uint8_t i = 0; i< CALLBACK_MAX;i++)
	  {
	    if(p_event_data->callbacks[i] == callback)
	    {
	    	p_event_data->callbacks[i] = NULL;
	    	p_event_data->registered_callbacks--;
	      return;
	    }
	  }
	  //LOGGER_ERROR("Callback not registered");
}

static void ClearVariable(void* data, uint8_t length, const void* defaultValue)
{
	if(defaultValue != NULL)
	{
	  memcpy(data,defaultValue, length);
	}
	else
	{
	  memset(data,0,length);
	}
}

static void LoadVariable(uint8_t* index, void* data, uint8_t length, uint16_t key, uint8_t multiply, const void* defaultValue)
{
	  int objectIndex = FlashRegisterObject(key, length, multiply);
	  if(objectIndex < 0)
	  {
	    //LOGGER_ERROR("Object could not be registered!");
	    return;
	  }
	  *index = (uint8_t)objectIndex;

	  bool written = false;
	  bool success = FlashReadObject(*index, (uint8_t*)data, length, &written);
	  if(!success)
	  {
	    //LOGGER_ERROR("Object could not be read!");
	    return;
	  }

	  if(!written)
	  {
		  if(defaultValue != NULL)
		  {
			  memcpy(data,defaultValue, length);
		  }
		  else
		  {
			  memset(data,0,length);
		  }
	    //LOGGER_INFO("Session Fixture Information never written --> Default");
	  }
	  else
	  {
	    //LOGGER_INFO("Session Fixture Information loaded");
	  }
}

static bool CheckDataChanged(uint8_t* data, uint8_t* compareData, int length)
{
  for(int i= 0; i< length; i++)
  {
    if(data[i] != compareData[i])
    {
      return true;
    }
  }
  return false;
}
