/**
  ******************************************************************************
  * @file           : system_implementation.c
  * @brief          : Simple example for showing overall system implementation when using iQ.Controller
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Kuenzler Technologies GbR
  * All rights reserved.
  *
  *
  ******************************************************************************
*/

/*==============================================================================*/
/* Includes                                                                  	*/
/*==============================================================================*/
#include "system_implementation.h"
#include "variables.h"
#include "libRdmInterface.h"
#include "rdm_implementation.h"
#include "measurement.h"
#include "runtime_implementation.h"
#include "dmx_512.h"
#include "counter.h"

#include "string.h"
#include "stdio.h"

/*==============================================================================*/
/* Defines & Macros                                                           	*/
/*==============================================================================*/

/*==============================================================================*/
/* Private functions                                                          	*/
/*==============================================================================*/
static void DMX512ReceivedRdmFrameCallback(const uint8_t* rdmData,uint16_t rdmLength);
static void DMX512ReceivedDmxFrameCallback(const uint8_t* dmxData,uint16_t dmxLength);

/* System Callbacks*/
static void SerialCallback(void);

/* Rdm  */
static void RdmAnswerCallback(const uint8_t *msg, uint16_t length, bool sentBreak, uint8_t id);

/*==============================================================================*/
/* Private variables                                                          	*/
/*==============================================================================*/
static GPIO_TypeDef* ledSignalPort = NULL;
static uint16_t ledSignalPin;
/*==============================================================================*/
/* Function definitions                                                     	*/
/*==============================================================================*/

/* Initialize Application (Callbacks,etc). */
void InitSystem(system_hal_config_t config)
{
	ledSignalPort = config.ledSignalPort;
	ledSignalPin = config.ledSignalPin;
	/* Initialize Variables, load flash,.. - 初始化变量，加载闪存*/
	InitVariables();
	/* Initialize RDM library and example implementation - 初始化RDM库和示例实现*/
	InitRdmImp(&RdmAnswerCallback);
	/* Initialize DMX512 Stack	*/
	InitDMX512(config.dmxHuart,config.dmxDirPort, config.dmxDdirPin,&DMX512ReceivedDmxFrameCallback, &DMX512ReceivedRdmFrameCallback);
	/* Initialize adc measurement implementation*/
	InitMeasurement();
	/* Initialize led driver implementation*/
	InitRuntimeImp(config.dimmer1PwmRegister, config.dimmer2PwmRegister, config.dimmer3PwmRegister, config.dimmer4PwmRegister,  config.dimmer5PwmRegister);

	InitCounter();

	/* Register System Callbacks - 注册系统回调*/
	register_notify_serial(&SerialCallback);
	/* update powercycle counter - 更新电源循环计数器*/
	IncrementPowercycle();
}

static uint32_t aliveCounter = 0;
#define ALIVECOUNT		10000

void SystemTask()
{
	uint32_t tickMs = HAL_GetTick();
	VariablesTask();
	DMX512_Task();
	RuntimeImpTask(tickMs);
	CounterTask(tickMs);

	// Show Alive
	if(aliveCounter + ALIVECOUNT < tickMs)
	{
		if(ledSignalPort != NULL)
		{
			HAL_GPIO_TogglePin(ledSignalPort, ledSignalPin);
		}
		aliveCounter = tickMs;
	}

}

/*==============================================================================*/
/* Private Function definitions                                                 */
/*==============================================================================*/

static void DMX512ReceivedDmxFrameCallback(const uint8_t* dmxData,uint16_t dmxLength)
{
	if(dmxLength >= RDM_FOOTPRINTS[*get_dmxPersonality() -1] + (*get_dmxStartadress() - 1))
	{
		switch(*get_dmxPersonality())
		{
		case 1:
			RuntimeImpSetDMXPersonaliy1(&dmxData[*get_dmxStartadress() -1]);
			break;
		default:
			break;
		}
	}
}

static void DMX512ReceivedRdmFrameCallback(const uint8_t* rdmData,uint16_t rdmLength)
{
	/* Always handle and answer RDM from all sources - 始终处理并回答所有来源的 RDM*/
	RdmSet(rdmData,rdmLength, 0);
}

/*==============================================================================*/
/* Callback functions                                                          	*/
/*==============================================================================*/

static void SerialCallback()
{
	/* Serialnumber changed. Recalculate RDMID and update all references - 序列号已更改。重新计算 RDMID 并更新所有引用 */
	rdmid_t id = get_rdmid();
	rdm_setRdmId(id.array);
}


/* Implementation from RDM Library. Provide the answer to the received message. 来自 RDM 库的实现。提供收到消息的答案。*/
static void RdmAnswerCallback(const uint8_t *msg, uint16_t length, bool sentBreak, uint8_t id)
{
	if(sentBreak)
	{
		DMX512_WriteRdmFrame(msg, length);
	}
	else
	{
		DMX512_WriteData(msg, length);
	}
	(void)id;
}

void SystemDimmerPeriodElapsedCallback(uint8_t dimmerIndex)
{
	RuntimeImpDimmerPeriodElapsedCallback(dimmerIndex);
}

/*==============================================================================*/
/* HAL Callback functions                                                          	*/
/*==============================================================================*/

void HAL_UART_TxCpltCallback(UART_HandleTypeDef *huart)
{
	DMX512_TxCpltCallback(huart);
}
void HAL_UART_ErrorCallback(UART_HandleTypeDef *huart)
{
	DMX512_ErrorCallback(huart);
}












