/**
  ******************************************************************************
  * @file           : led_driver.c
  * @brief          : Led driver which control powerstage, current and dimming
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Kuenzler Technologies GbR
  * All rights reserved.
  *
  *
  ******************************************************************************
*/

/*==============================================================================*/
/* Includes                                                                  	*/
/*==============================================================================*/
#include "runtime_implementation.h"
#include "variables.h"
#include "math.h"
#include "measurement.h"
#include "string.h"
#include "defines.h"

/*==============================================================================*/
/* Defines & Macros                                                           	*/
/*==============================================================================*/
// The SUM of the power multiplied with ledCount should not be higher than 65535
static const uint16_t POWER_CONSUMPTION_DIE[DIE_COUNT] = {2400,3000,2600,2600, 5100};
static const uint16_t BATTERYENERGYH = 16000;
static const uint16_t TIMECONSTANT = 50;
/*==============================================================================*/
/* Private Variables                                                         	*/
/*==============================================================================*/
static volatile uint32_t *dimmer1;
static volatile uint32_t *dimmer2;
static volatile uint32_t *dimmer3;
static volatile uint32_t *dimmer4;
static volatile uint32_t *dimmer5;

static runtime_instance_t runtime;
/*==============================================================================*/
/* Private Function declarations                                                 */
/*==============================================================================*/
static uint16_t bswap16(uint16_t val);
/*==============================================================================*/
/* Function definitions                                                     	*/
/*==============================================================================*/

void InitRuntimeImp(volatile uint32_t *dimmer1Register, volatile uint32_t *dimmer2Register, volatile uint32_t *dimmer3Register, volatile uint32_t *dimmer4Register, volatile uint32_t *dimmer5Register)
{
	dimmer1 = dimmer1Register;
	dimmer2 = dimmer2Register;
	dimmer3 = dimmer3Register;
	dimmer4 = dimmer4Register;
	dimmer5 = dimmer5Register;

	InitRuntime(&runtime, LED_COUNT, DIE_COUNT, (uint16_t*)POWER_CONSUMPTION_DIE, BATTERYENERGYH, TIMECONSTANT);
	SetRuntime(&runtime,BATTERYRUNTIME_20HOURS,5000);
}

void RuntimeImpTask(uint32_t tickMs)
{
	(void)tickMs;
}

void RuntimeImpSetBatteryRuntime(uint8_t runtimeHours)
{
	SetRuntime(&runtime,(battery_runtime_t)runtimeHours,0xFFFF);
}

void RuntimeImpSetDMXPersonaliy1(const uint8_t* data)
{
	// Master Dimmer Coarse Fine
	uint16_t mDimmer;
	memcpy(&mDimmer, &data[0], sizeof(uint16_t));
	mDimmer = bswap16(mDimmer);

	uint8_t mStrobe;
	memcpy(&mStrobe, &data[2], sizeof(uint8_t));

	uint16_t red;
	memcpy(&red, &data[3], sizeof(uint16_t));
	red = bswap16(red);

	uint16_t green;
	memcpy(&green, &data[5], sizeof(uint16_t));
	green = bswap16(green);

	uint16_t blue;
	memcpy(&blue, &data[7], sizeof(uint16_t));
	blue = bswap16(blue);

	uint16_t amber;
	memcpy(&amber, &data[9], sizeof(uint16_t));
	amber = bswap16(amber);

	uint16_t lime;
	memcpy(&lime, &data[11], sizeof(uint16_t));
	lime = bswap16(lime);

	uint16_t array[LED_COUNT][DIE_COUNT];
	for(uint8_t led=0; led < LED_COUNT;led++)
	{
		array[led][0] = red;
		array[led][1] = green;
		array[led][2] = blue;
		array[led][3] = lime;
		array[led][4] = amber;
	}
	DoRuntimeSelectionCalculation16Bit(&runtime, array[0]);

	// Red
	*dimmer3 = array[0][0];
	// Green
	*dimmer4 = array[0][1];
	// Blue
	*dimmer2 = array[0][2];
	// Amber
	*dimmer5 = array[0][4];
	// Lime
	*dimmer1 = array[0][3];
}

bool RuntimeImpLightisOn(void)
{
	return *dimmer1 > 0 || *dimmer2 > 0 || *dimmer3 > 0 || *dimmer4 > 0 || *dimmer5 > 0;
}

void RuntimeImpDimmerPeriodElapsedCallback(uint8_t dimmerIndex)
{

}

/*==============================================================================*/
/* Private Function definitions                                                	*/
/*==============================================================================*/


static uint16_t bswap16(uint16_t val)
{
	 return ((val & 0xFF) << 8) | (val >> 8);
}

/*==============================================================================*/
/* Callback functions                                                          	*/
/*==============================================================================*/



