/**
  ******************************************************************************
  * @file           : rdm_implementation.c
  * @brief          : Simple example for showing rdm functionality using the provided rdm library
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Kuenzler Technologies GbR
  * All rights reserved.
  *
  *
  ******************************************************************************
*/

/*==============================================================================*/
/* Includes                                                                  	*/
/*==============================================================================*/
#include "rdm_implementation.h"
#include "string.h"
#include "stdio.h"
#include "variables.h"
#include "defines.h"
#include "libRdmInterface.h"
#include "measurement.h"
#include "runtime_implementation.h"


/*==============================================================================*/
/* Private functions                                                          	*/
/*==============================================================================*/
static uint8_t dump_u64(uint64_t x, char* buf, uint8_t bufLength);
/*==============================================================================*/
/* Function definitions                                                     	*/
/*==============================================================================*/
void InitRdmImp(void (*callback)(const uint8_t * msg, uint16_t length,bool sendBreak, uint8_t id))
{
	rdm_init(get_rdmid().array,callback);
}

void RdmSet(const uint8_t* buf, uint16_t length, uint8_t id)
{
	rdm_decodeMessage(buf, length, id);
}

void RdmExtGetFactoryDefaults(rdm_answer_t* rdmImp,rdm_factory_default_get_response_t* responseBuf )
{
	responseBuf->isDefault = get_factoryDefault();
	(void)rdmImp;
}

void RdmExtGetDeviceInfo(rdm_answer_t* rdmImp, rdm_device_info_get_response_t* responseBuf)
{
	responseBuf->SubDeviceCount = 0;
	responseBuf->deviceModelId = RDM_DEVICEMODEL_ID;
	responseBuf->dmxFootprint = RDM_FOOTPRINTS[*get_dmxPersonality() -1];
	responseBuf->dmxPersonality = *get_dmxPersonality();
	responseBuf->dmxPersonalityCount = RDM_PERSONALITY_COUNT;
	responseBuf->dmxStartaddress = *get_dmxStartadress();
	responseBuf->productCategory = RDM_PRODUCT_CATEGORY;//PRODUCT_CATEGORY_FIXTURE_FIXED;
	responseBuf->protocollVersion = 256;
	responseBuf->sensorCount = RDM_SENSOR_COUNT;
	responseBuf->softwareVersionId = (VERSIONMAJOR << 24) + (VERSIONMINOR << 16) + (VERSIONPATCH << 8) + (VERSIONRELEASE);
	(void)rdmImp;
}
void RdmExtGetProductDetail(rdm_answer_t* rdmImp, rdm_product_detail_id_get_response_t* responseBuf)
{
	responseBuf->productDetails[0] = RDM_PRODUCT_DETAIL;
	responseBuf->productDetailsCount = 1;
	(void)rdmImp;
}
void RdmExtGetDeviceModelDescription(rdm_answer_t* rdmImp, rdm_device_model_description_get_response_t* responseBuf)
{
	strcpy((char*)responseBuf->label,RDM_DEVICE_MODEL_DESCRIPTION);
	responseBuf->labelSize = sizeof(RDM_DEVICE_MODEL_DESCRIPTION);
	(void)rdmImp;
}
void RdmExtGetManufacturerLabel(rdm_answer_t* rdmImp, rdm_manufacturer_label_get_response_t* responseBuf)
{
	strcpy((char*)responseBuf->label,RDM_MANUFACTURER_NAME);
	responseBuf->labelSize = sizeof(RDM_MANUFACTURER_NAME);
	(void)rdmImp;
}
void RdmExtGetDeviceLabel(rdm_answer_t* rdmImp, rdm_device_label_get_response_t* responseBuf)
{
	device_label_t label = *get_deviceLabel();
	uint8_t length = (uint8_t)strlen(label.array);
	memcpy(responseBuf->label,label.array,length);
	responseBuf->labelSize = length;
	(void)rdmImp;
}
void RdmExtGetSoftwareVersionLabel(rdm_answer_t* rdmImp,rdm_softwareversion_label_get_response_t* responseBuf)
{
	responseBuf->labelSize = (uint8_t)sprintf((char*)responseBuf->label, "%d.%d.%d.%d", VERSIONMAJOR,VERSIONMINOR,VERSIONPATCH,VERSIONRELEASE);
	(void)rdmImp;
}
void RdmExtGetBootSoftwareVersionLabel(rdm_answer_t* rdmImp, rdm_boot_softwareversion_label_get_response_t* responseBuf)
{
	responseBuf->labelSize = (uint8_t)sprintf((char*)responseBuf->label, "%d.%d.%d.%d", 0,0,0,0);
	(void)rdmImp;
}
void RdmExtGetDmxPersonality(rdm_answer_t* rdmImp, rdm_dmx_personality_get_response_t* responseBuf)
{
	responseBuf->currentPersonality = *get_dmxPersonality();
	responseBuf->numPersonalities = RDM_PERSONALITY_COUNT;
	(void)rdmImp;
}
void RdmExtGetDmxPersonalityDescription(rdm_answer_t* rdmImp, rdm_dmx_personality_description_get_response_t* responseBuf)
{
	if(responseBuf->requestedPersonality > RDM_PERSONALITY_COUNT)
	{
		 rdmImp->response = RDM_NACK_REASON;
		 rdmImp->reason = NR_DATA_OUT_OF_RANGE;
		 return;
	}
	responseBuf->footprint = RDM_FOOTPRINTS[responseBuf->requestedPersonality -1];
	responseBuf->labelSize = (uint8_t)strlen(RDM_PERSONALITY_NAMES[responseBuf->requestedPersonality -1]);
	memcpy((char*)responseBuf->label,RDM_PERSONALITY_NAMES[responseBuf->requestedPersonality -1],responseBuf->labelSize);
}
void RdmExtGetDmxStartaddress(rdm_answer_t* rdmImp, rdm_dmx_startaddress_get_response_t* responseBuf)
{
	responseBuf->address = *get_dmxStartadress();
	(void)rdmImp;
}
void RdmExtGetSensorDefinition(rdm_answer_t* rdmImp, rdm_sensor_defintion_get_response_t* responseBuf)
{
	/* 	0 -  tempNtc;
		1 -  tempPcb;
		2 -  inputVoltage;
		3 -  outputVoltage;
		4 -  outputCurrent;
		5 -  outputPower; */
	switch(responseBuf->requestedSensor)
	{
	case 0:	// tempNtc
		responseBuf->labelSize = (uint8_t)sprintf((char*)responseBuf->label, "External NTC Temperature");
		responseBuf->normalMaximumValue = 1500;
		responseBuf->normalMinimumValue = -400;
		responseBuf->prefix = RDM_PREFIX_DECI;
		responseBuf->rangeMaximumValue = 1500;
		responseBuf->rangeMaximumValue = -400;
		responseBuf->recordedValueSupport = false;
		responseBuf->type = RDMSENS_TEMPERATURE;
		responseBuf->unit = RDMUNITS_CENTIGRADE;
		break;
	case 1:	// tempIc
			responseBuf->labelSize = (uint8_t)sprintf((char*)responseBuf->label, "IC Temperature");
			responseBuf->normalMaximumValue = 1300;
			responseBuf->normalMinimumValue = 0;
			responseBuf->prefix = RDM_PREFIX_DECI; // PREFIX_NONE
			responseBuf->rangeMaximumValue = 1300;
			responseBuf->rangeMaximumValue = 0;
			responseBuf->recordedValueSupport = false;
			responseBuf->type = RDMSENS_TEMPERATURE;
			responseBuf->unit = RDMUNITS_CENTIGRADE;
			break;
	default:
		 rdmImp->response = RDM_NACK_REASON;
		 rdmImp->reason = NR_DATA_OUT_OF_RANGE;
		break;
	}

}
void RdmExtGetSensorValue(rdm_answer_t* rdmImp, rdm_sensor_value_get_response_t* responseBuf)
{
	/* 	0 -  tempNtc;
		1 -  tempPcb;
		2 -  inputVoltage;
		3 -  outputVoltage;
		4 -  outputCurrent;
		5 -  outputPower; */
	switch(responseBuf->requestedSensor)
	{
		case 0:	// tempNtc
			responseBuf->presentValue = (uint16_t)(Measurement_GetTemperatureNtc() * 10);
			responseBuf->highestDetectedValue = 0;
			responseBuf->lowestDetectedValue = 0;
			responseBuf->recordedValue = 0;
		break;
		case 1:	// tempIc
			responseBuf->presentValue = (uint16_t)(Measurement_GetIcTemperature() * 10);
			responseBuf->highestDetectedValue = 0;
			responseBuf->lowestDetectedValue = 0;
			responseBuf->recordedValue = 0;
		break;
		default:
			 rdmImp->response = RDM_NACK_REASON;
			 rdmImp->reason = NR_DATA_OUT_OF_RANGE;
		break;
	}
}
void RdmExtGetDeviceHours(rdm_answer_t* rdmImp, rdm_device_hours_get_response_t* responseBuf)
{
  responseBuf->deviceHours = *get_deviceMinutes()/60;
  (void)rdmImp;
}
void RdmExtGetLampHours(rdm_answer_t* rdmImp, rdm_lamphours_get_response_t* responseBuf)
{
	responseBuf->lampHours = *get_sourceMinutes()/60;
	(void)rdmImp;
}
void RdmExtGetLampStrikes(rdm_answer_t* rdmImp, rdm_lampstrike_get_response_t* responseBuf)
{
   rdmImp->response = RDM_NACK_REASON;
   (void)responseBuf;
}
void RdmExtGetLampState(rdm_answer_t* rdmImp, rdm_lampstate_get_response_t* responseBuf)
{
   rdmImp->response = RDM_NACK_REASON;
   (void)responseBuf;
}
void RdmExtGetDevicePowerCycles(rdm_answer_t* rdmImp, rdm_device_power_cycle_get_response_t* responseBuf)
{
  responseBuf->powerCycleCount = *get_powerCycles();
  (void)rdmImp;
}
void RdmExtGetIdentify(rdm_answer_t* rdmImp, rdm_identify_device_get_response_t* responseBuf)
{
	responseBuf->identifyState = *get_rdmIdentify();
	(void)rdmImp;
}
void RdmExtGetPowerState(rdm_answer_t* rdmImp, rdm_power_state_get_response_t* responseBuf)
{
	responseBuf->powerState = RDMPOWER_STATE_NORMAL;
	(void)rdmImp;
}

void RdmExtGetPerformSelftest(rdm_answer_t* rdmImp, rdm_perform_selftest_get_response_t* responseBuf)
{
  rdmImp->response = RDM_NACK_REASON;
  (void)responseBuf;
}
void RdmExtGetSelfTestDescription(rdm_answer_t* rdmImp,rdm_selftest_description_get_response_t* responseBuf)
{
  rdmImp->response = RDM_NACK_REASON;
  (void)responseBuf;
}


void RdmExtGetSlotInfo(rdm_answer_t* rdmImp,rdm_slot_info_get_response_array_t* responseBuf, uint8_t* slotCount)
{
	static uint16_t sentSlotCount = 0;
	uint8_t i;
	for(i=0; i < RDM_MAX_SLOTINFO;i++)
	{
		if(i + sentSlotCount >= RDM_FOOTPRINTS[*get_dmxPersonality() - 1])
		{
			break;
		}
		responseBuf[i] = RDM_SLOT_INFO[*get_dmxPersonality() - 1][sentSlotCount + i];
	}
	if(i < RDM_MAX_SLOTINFO -1)
	{
		rdmImp->response = RDM_ACK;
		sentSlotCount = 0;
	}
	else
	{
		rdmImp->response = RDM_ACK_OVERFLOW;
		sentSlotCount += i;
	}
	*slotCount = i;

}

void RdmExtGetSlotDescription(rdm_answer_t* rdmImp,rdm_slot_desc_get_response_t* responseBuf)
{
	if(responseBuf->requestedSlot >= RDM_FOOTPRINTS[*get_dmxPersonality() -1])
	{
		rdmImp->response = RDM_NACK_REASON;
		rdmImp->reason = NR_DATA_OUT_OF_RANGE;
		return;
	}
	responseBuf->labelSize = (uint8_t)strlen(RDM_PERSONALITY_CHANNEL_NAMES[*get_dmxPersonality() -1][responseBuf->requestedSlot]);
	memcpy(responseBuf->description, RDM_PERSONALITY_CHANNEL_NAMES[*get_dmxPersonality() -1][responseBuf->requestedSlot], responseBuf->labelSize);
}

void RdmExtGetDefaultSlotValue(rdm_answer_t* rdmImp,rdm_slot_default_get_response_array_t* responseBuf, uint8_t* slotCount)
{
	*slotCount = 2;
	// Channel 1
	responseBuf[0].slotOffset = 0;
	responseBuf[0].slotDefaultValue = 0;
	// Channel 2
	responseBuf[0].slotOffset = 1;
	responseBuf[0].slotDefaultValue = 0;
	(void)rdmImp;
}

void RdmExtGetIq(rdm_answer_t* rdmImp, rdm_iq_get_response_t* responseBuf)
{
	uint8_t warningCount = 0;
	uint8_t errorCount = 0;
	responseBuf->protocollVersion = 1;
	responseBuf->deviceModelId = RDM_DEVICEMODEL_ID;
	responseBuf->serial = (uint64_t)*get_serial();

	// No bootloader or firmware version available - Anounce it as 0.0.1.255
	responseBuf->fixtureFirmware[0] = VERSIONMAJOR;
	responseBuf->fixtureFirmware[1] = VERSIONMINOR;
	responseBuf->fixtureFirmware[2] = VERSIONPATCH;
	responseBuf->fixtureFirmware[3] = VERSIONRELEASE;
	memcpy(responseBuf->fixtureMainCpuId, (uint8_t*) 0x1FFFF7AC, 12);
	responseBuf->battery = 0;
	responseBuf->powerSupplyState = IQ_POWERSUPPLY_STATE_GRID_POWERED;
	responseBuf->inputSource = RDMIQINPUTSOURCE_DMX;
	responseBuf->dmxFootprint = RDM_FOOTPRINTS[*get_dmxPersonality() -1];
	responseBuf->dmxPersonality = *get_dmxPersonality();
	responseBuf->dmxStartaddress = *get_dmxStartadress();
    
	if(warningCount > 15)
	{
		warningCount = 15;
	}
   
	if(errorCount > 15)
	{
		errorCount = 15;
	}
	responseBuf->errorCount = errorCount;
	responseBuf->warningCount = warningCount;

	responseBuf->supportedInputSource = (1 << 1); // DMX Support only

	(void)rdmImp;
}

void RdmExtGetCustomError(rdm_answer_t* rdmImp, rdm_custom_label_get_response_t* responseBuf)
{
	uint8_t counter = 0;
//	if(LedDriverInputVoltageError())
//	{
//		strcpy((char*)&(responseBuf->label[counter]),"UVLOCKOUT ");
//		counter += sizeof("UVLOCKOUT ");
//	}
//	if(LedDriverOverTemperatureError())
//	{
//		strcpy((char*)&(responseBuf->label[counter]),"OVERTEMP ");
//		counter += sizeof("OVERTEMP ");
//	}
	responseBuf->labelSize = counter;
	(void)rdmImp;
}

void RdmExtGetCustomWarning(rdm_answer_t* rdmImp, rdm_custom_label_get_response_t* responseBuf)
{
	uint8_t counter = 0;
	responseBuf->labelSize = counter;
	(void)rdmImp;
}

void RdmExtGetCustomServiceHours(rdm_answer_t* rdmImp, uint32_t* responseBuf)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNSUPPORTED_COMMAND_CLASS;
	(void)responseBuf;
}

void RdmExtSetFactoryDefaults(rdm_answer_t* rdmImp)
{
  set_factoryDefault();
  (void)rdmImp;
}
void RdmExtSetDeviceLabel(rdm_answer_t* rdmImp, char* label, uint8_t labelLength)
{
	device_label_t dlabel;
	memset(&dlabel,0, sizeof(device_label_t));
    memcpy(dlabel.array,label,labelLength);
    set_deviceLabel(dlabel);
    (void)rdmImp;
}
void RdmExtSetDmxPersonality(rdm_answer_t* rdmImp, uint8_t personality)
{
  if(!set_dmxPersonality(personality))
  {
    rdmImp->response = RDM_NACK_REASON;
    rdmImp->reason = NR_DATA_OUT_OF_RANGE;
  }
}
void RdmExtSetDmxStartaddress(rdm_answer_t* rdmImp,uint16_t dmxStartAddress)
{
	set_dmxStartadress(dmxStartAddress);
	(void)rdmImp;
	(void)dmxStartAddress;
}
void RdmExtSetIdentify(rdm_answer_t* rdmImp, bool set)
{
	 set_rdmIdentify(set);
	 (void)rdmImp;
	 (void)set;
}
void RdmExtSetResetDevice(rdm_answer_t* rdmImp,rdm_reset_t reset)
{
	 rdmImp->response = RDM_NACK_REASON;
	 rdmImp->reason = NR_UNSUPPORTED_COMMAND_CLASS;
	 (void)reset;
}
void RdmExtSetPowerState(rdm_answer_t* rdmImp, rdm_powerstate_t _powerState)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNSUPPORTED_COMMAND_CLASS;
	(void)_powerState;
}
void RdmExtSetPerformSelftest(rdm_answer_t* rdmImp,uint8_t selftest)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNSUPPORTED_COMMAND_CLASS;
	(void)selftest;
}

void RdmExtSetCustomServiceHours(rdm_answer_t* rdmImp, uint32_t counter)
{
	rdmImp->response = RDM_NACK_REASON;
	rdmImp->reason = NR_UNSUPPORTED_COMMAND_CLASS;
	(void)counter;
}

void RdmExtSetColorConverterLicense(rdm_answer_t* rdmImp,rdm_colorconverter_license_t key)
{
	cc_key_store_t data;
	memcpy(data.array,key.licenseData,sizeof(data.array));
	set_ccKey(data);
	(void)rdmImp;
}

void RdmExtSetColorConverterCalibration(rdm_answer_t* rdmImp,rdm_colorconverter_cal_command_t command, uint16_t dataOffset, uint16_t* calData, uint8_t length)
{
	if((dataOffset + length) * 2 > sizeof(cc_calibration_store_t))
	{
		rdmImp->response = RDM_NACK_REASON;
		rdmImp->reason = NR_FORMAT_ERROR;
		return;
	}
	cc_calibration_store_t data = *get_ccCal();
	memcpy(&(data.array) + dataOffset * 2,calData,length * 2);
	set_ccCal(data);
	(void)command;
	(void)rdmImp;
}


typedef enum
{
	RDMCPID_CUSTOMSERIAL = 0x8060,

	RDMCPID_CUSTOM_ENABLEDRIVER 	= 0x9020,
	RDMCPID_CUSTOM_CURRENT   		= 0x9021,
	RDMCPID_CUSTOM_DIMMER    		= 0x9022,
	RDMCPID_CUSTOM_SHUNTRESISTANCE  = 0x9023,
	RDMCPID_CUSTOM_ENABLEMEASUREC  	= 0x9024,
	RDMCPID_CUSTOM_MEASURECURRENT	= 0x9025,
	RDMCPID_CUSTOM_PCB_SERIAL		= 0x9028,
	RDM_CPID_CUSTOM_CC_INFO			= 0x9029,
}rdm_custom_pid_t;

void RdmExtGetSupportedParameter(rdm_answer_t* rdmImp,uint16_t* responseBuf, uint8_t* pidCount)
{
	responseBuf[0] = RDMPID_PRODUCTDETAILIDLIST;
	responseBuf[1] = RDMPID_DEVICEMODELDESCRIPTION;
	responseBuf[2] = RDMPID_MANUFACTURER_LABEL;
	responseBuf[3] = RDMPID_DEVICE_LABEL;
	responseBuf[4] = RDMPID_FACTORY_DEFAULTS;
	responseBuf[5] = RDMPID_BOOT_SOFTWARE_VERSION_LABEL;
	responseBuf[6] = RDMPID_DMX_PERSONALITY;
	responseBuf[7] = RDMPID_DMX_PERSONALITY_DESCRIPTION;
	responseBuf[8] = RDMPID_SLOT_INFO;
	responseBuf[9] = RDMPID_SLOT_DESCRIPTION;
	responseBuf[10] = RDMPID_SENSOR_DEFINITION;
	responseBuf[11] = RDMPID_SENSOR_VALUE;
	responseBuf[12] = RDMPID_DEVICE_HOURS;
	responseBuf[13] = RDMPID_LAMP_HOURS;
	responseBuf[14] = RDMPID_DEVICE_POWER_CYCLES;
	responseBuf[15] = RDMPID_RESET_DEVICE;
	responseBuf[16] = RDMPID_POWER_STATE;
	responseBuf[17] = RDMPID_PERFORM_SELFTEST;
	responseBuf[18] = RDMCPID_CUSTOMERROR;
	responseBuf[19] = RDMCPID_CUSTOMWARNING;
	responseBuf[20] = RDMCPID_CUSTOMSERIAL;
	responseBuf[21] = RDMCPID_CUSTOM_PCB_SERIAL;
	*pidCount = 22;
	(void)rdmImp;
}

void RdmExtGetParameterDescription(rdm_answer_t* rdmImp,rdm_parameter_description_get_response_t* responseBuf)
{
	//Command Class Defines Value Description
	//CC_GET 0x01 PID supports GET only
	//CC_SET 0x02 PID supports SET only
	//CC_GET_SET 0x03 PID supports GET & SET
    switch(responseBuf->requestedParameter)
    {
    	case RDMCPID_CUSTOMERROR:
    		responseBuf->commandClass = RDM_PD_CC_GET;
    		responseBuf->dataType = RDM_DTD_DS_ASCII;
			responseBuf->defaultValue = 0;
			responseBuf->labelSize = (uint8_t)sprintf((char*)responseBuf->description, "Error Codes");
			responseBuf->maxValidValue = 0;
			responseBuf->pdlSize = 32;
			responseBuf->prefix = RDM_PREFIX_NONE;
			responseBuf->type = 0x00;
			responseBuf->unit = RDMUNITS_NONE;

    	break;
    	case RDMCPID_CUSTOMWARNING:
    		responseBuf->commandClass = RDM_PD_CC_GET;
    		responseBuf->dataType = RDM_DTD_DS_ASCII;
			responseBuf->defaultValue = 0;
			responseBuf->labelSize = (uint8_t)sprintf((char*)responseBuf->description, "Warning Codes");
			responseBuf->maxValidValue = 0;
			responseBuf->pdlSize = 32;
			responseBuf->prefix = RDM_PREFIX_NONE;
			responseBuf->type = 0x00;
			responseBuf->unit = RDMUNITS_NONE;
    	break;
    	case RDMCPID_CUSTOMSERIAL:
    		responseBuf->commandClass = RDM_PD_CC_GET;
			responseBuf->dataType = RDM_DTD_DS_ASCII;
			responseBuf->defaultValue = 0;
			responseBuf->labelSize = (uint8_t)sprintf((char*)responseBuf->description, "Serial number");
			responseBuf->maxValidValue = 0;
			responseBuf->pdlSize = 32;
			responseBuf->prefix = RDM_PREFIX_NONE;
			responseBuf->type = 0x00;
			responseBuf->unit = RDMUNITS_NONE;
    	break;
    	case RDMCPID_CUSTOM_ENABLEDRIVER:
    		responseBuf->commandClass = RDM_PD_CC_GET_SET;
			responseBuf->dataType = RDM_DTD_DS_UNSIGNED_BYTE;
			responseBuf->defaultValue = 0;
			responseBuf->labelSize = (uint8_t)sprintf((char*)responseBuf->description, "Driver Enable");
			responseBuf->maxValidValue = 1;
			responseBuf->pdlSize = 1;
			responseBuf->prefix = RDM_PREFIX_NONE;
			responseBuf->type = 0x00;
			responseBuf->unit = RDMUNITS_NONE;
    	break;
       	case RDMCPID_CUSTOM_CURRENT:
       		responseBuf->commandClass = RDM_PD_CC_GET_SET;
			responseBuf->dataType = RDM_DTD_DS_UNSIGNED_WORD;
			responseBuf->defaultValue = 0;
			responseBuf->labelSize = (uint8_t)sprintf((char*)responseBuf->description, "Driver Current");
			responseBuf->maxValidValue = 4000;
			responseBuf->pdlSize = 2;
			responseBuf->prefix = RDM_PREFIX_MILLI;
			responseBuf->type = 0x00;
			responseBuf->unit = RDMUNITS_AMPERE_DC;
		break;
       	case RDMCPID_CUSTOM_DIMMER:
       		responseBuf->commandClass = RDM_PD_CC_GET_SET;
			responseBuf->dataType = RDM_DTD_DS_UNSIGNED_WORD;
			responseBuf->defaultValue = 0;
			responseBuf->labelSize = (uint8_t)sprintf((char*)responseBuf->description, "Driver Dimmer");
			responseBuf->maxValidValue = 65535;
			responseBuf->pdlSize = 2;
			responseBuf->prefix = RDM_PREFIX_NONE;
			responseBuf->type = 0x00;
			responseBuf->unit = RDMUNITS_NONE;
		break;
    	case RDMCPID_CUSTOM_SHUNTRESISTANCE:
           		responseBuf->commandClass = RDM_PD_CC_GET_SET;
    			responseBuf->dataType = RDM_DTD_DS_UNSIGNED_WORD;
    			responseBuf->defaultValue = 0;
    			responseBuf->labelSize = (uint8_t)sprintf((char*)responseBuf->description, "Driver Shunt Resistance");
    			responseBuf->maxValidValue = 100;
    			responseBuf->pdlSize = 2;
    			responseBuf->prefix = RDM_PREFIX_MILLI;
    			responseBuf->type = 0x00;
    			responseBuf->unit = RDMUNITS_OHM;
    		break;
    	case RDMCPID_CUSTOM_PCB_SERIAL:
			responseBuf->commandClass = RDM_PD_CC_GET_SET;
			responseBuf->dataType = RDM_DTD_DS_UNSIGNED_WORD;
			responseBuf->defaultValue = 0;
			responseBuf->labelSize = (uint8_t)sprintf((char*)responseBuf->description, "PCB Serial");
			responseBuf->maxValidValue = 100;
			responseBuf->pdlSize = 2;
			responseBuf->prefix = RDM_PREFIX_NONE;
			responseBuf->type = 0x00;
			responseBuf->unit = RDMUNITS_NONE;
    		break;
    	case RDMCPID_CUSTOM_ENABLEMEASUREC:
			responseBuf->commandClass = RDM_PD_CC_GET_SET;
			responseBuf->dataType = RDM_DTD_DS_UNSIGNED_BYTE;
			responseBuf->defaultValue = 0;
			responseBuf->labelSize = (uint8_t)sprintf((char*)responseBuf->description, "Enable Measure Current");
			responseBuf->maxValidValue = 1;
			responseBuf->pdlSize = 1;
			responseBuf->prefix = RDM_PREFIX_NONE;
			responseBuf->type = 0x00;
			responseBuf->unit = RDMUNITS_NONE;
			break;
      default:
         rdmImp->response = RDM_NACK_REASON;
         rdmImp->reason = NR_DATA_OUT_OF_RANGE;
      break;
    }
}

/*==============================================================================*/
/* Czstom PIDs                                                         			*/
/*==============================================================================*/
typedef struct  __attribute__((__packed__))
{
	bool enabled;
}rdm_custom_driver_enable_set_get_t;

typedef struct  __attribute__((__packed__))
{
	uint16_t currentmA;
}rdm_custom_driver_current_set_get_t;

typedef struct  __attribute__((__packed__))
{
	uint16_t dimmer;
}rdm_custom_driver_dimmer_set_get_t;

typedef struct  __attribute__((__packed__))
{
	bool enabled;
}rdm_custom_measurement_enable_set_get_t;

typedef struct  __attribute__((__packed__))
{
	uint8_t type;
	uint16_t length;
}rdm_custom_measure_current_set_t;

typedef struct  __attribute__((__packed__))
{
	uint16_t offset;
	uint16_t length;
}rdm_custom_measure_current_get_t;

static void DecodeSetCustomPcbSerial(rdm_answer_t* rdmImp, uint8_t * data, uint16_t length);

static void DecodeGetCustomSerial(rdm_answer_t* rdmImp, uint8_t* responseBuf, uint16_t* responseLength);

static void DecodeGetCustomPcbSerial(rdm_answer_t* rdmImp, uint8_t* responseBuf, uint16_t* responseLength);

void RdmExtSetCustomPid(rdm_answer_t* rdmImp, uint16_t pid, uint8_t* data, uint16_t pdl)
{
  switch(pid)
  {
      case RDMCPID_CUSTOM_PCB_SERIAL:
		  DecodeSetCustomPcbSerial(rdmImp,data,pdl);
		break;
      default:
        rdmImp->response = RDM_NACK_REASON;
        rdmImp->reason = NR_UNKNOWN_PID;
      break;
  }
}
void RdmExtGetCustomPid(uint16_t pid, uint8_t* msg, uint16_t pdl, rdm_answer_t* rdmImp, uint8_t* responseBuf, uint16_t* responseLength)
{
  switch(pid)
  {
      case RDMCPID_CUSTOMSERIAL:
    	  DecodeGetCustomSerial(rdmImp,responseBuf,responseLength);
      break;
      case RDMCPID_CUSTOM_PCB_SERIAL:
    	  DecodeGetCustomPcbSerial(rdmImp,responseBuf,responseLength);
      	  break;
      default:
        rdmImp->response = RDM_NACK_REASON;
        rdmImp->reason = NR_UNKNOWN_PID;
      break;
  }
  (void)msg;
  (void)pdl;
}

static void DecodeGetCustomSerial(rdm_answer_t* rdmImp, uint8_t* responseBuf, uint16_t* responseLength)
{
  uint64_t serial = *get_serial();
  *responseLength = dump_u64(serial,(char*)responseBuf,32);
  (void) rdmImp;
}


static void DecodeGetCustomPcbSerial(rdm_answer_t* rdmImp, uint8_t* responseBuf, uint16_t* responseLength)
{
	rdm_custom_driver_dimmer_set_get_t* response = (rdm_custom_driver_dimmer_set_get_t*)responseBuf;
	response->dimmer = (uint16_t)(*get_serial());
	response->dimmer = __builtin_bswap16(response->dimmer);
	*responseLength = sizeof(rdm_custom_driver_dimmer_set_get_t);
	(void)rdmImp;
}

static void DecodeSetCustomPcbSerial(rdm_answer_t* rdmImp, uint8_t * data, uint16_t length)
{
	if(length >= sizeof(rdm_custom_driver_dimmer_set_get_t))
	{
		rdm_custom_driver_dimmer_set_get_t* set = (rdm_custom_driver_dimmer_set_get_t*)data;
		uint16_t serial = (__builtin_bswap16(set->dimmer));
		set_serial(serial);
	}
	else
	{
		rdmImp->response = RDM_NACK_REASON;
		rdmImp->reason = NR_FORMAT_ERROR;
	}
}

static uint8_t dump_u64(uint64_t x, char* buf, uint8_t bufLength)
{
	char temp[bufLength];
	char* b = temp;
	uint8_t count = 0;
    do
    {
    	if(count < bufLength)
    	{
    		*b++ = (char)('0' + (x % 10));
    		count++;
    	}
    	x /= 10;
    } while (x);
    for(uint8_t i= 0; i< count;i++)
    {
    	buf[i] = temp[count -1 - i];
    }
    return count;
}
