/**
  ******************************************************************************
  * @file           : adc.c
  * @brief          : Simple example for reading analog voltage
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Kuenzler Technologies GbR
  * All rights reserved.
  *
  *
  ******************************************************************************
*/

/*==============================================================================*/
/* Includes                                                                  	*/
/*==============================================================================*/

#include "measurement.h"
#include "main.h"
#include "variables.h"
#include "math.h"
#include "string.h"

/*==============================================================================*/
/* Defines & Macros                                                           	*/
/*==============================================================================*/

#define VREF				    		((float)2.5f)
#define ADCRESOLUTION					4095
#define ADCOVERSAMPLERESOLTION			0xFFFF

#define INPUTVOLTAGE_RESISTOR1			47000
#define INPUTVOLTAGE_RESISTOR2	 		2200

#define NTC_RESISTOR1	 				10000
#define NTC_NOMINAL_R	 				10000
#define NTC_NOMINAL_TEMP        		25

#define PCB_TEMP_SENSOR_V0				0.4f
#define PCB_TEMP_SENSOR_TC				0.0195f

#define OUTPUTVOLTAGE_RESISTOR1			47000
#define OUTPUTVOLTAGE_RESISTOR2	 		2700

#define CURRENTVOLTAGE_GAIN				20

#define SAMPLEWINDOW1					1024
#define SAMPLEWINDOW2					200
#define LOGRESOLUTION					93


//#define ADC4_CLOCK_FREQ_KHZ				42500
//#define ADC4_SAMPLINGCYCLE				(2.5f + 12.5f)
//#define ADC4_SAMPLINGFREQUENCY_KHZ	    (ADC4_CLOCK_FREQ_KHZ / ADC4_SAMPLINGCYCLE)
/*==============================================================================*/
/* Private Variables                                                         	*/
/*==============================================================================*/

static uint16_t dataAdc2[3];

static uint16_t dataAdc5[2];



/*==============================================================================*/
/* Function definitions                                                     	*/
/*==============================================================================*/
void InitMeasurement()
{
		// Calibrate The ADC On Power-Up For Better Accuracy
		HAL_ADCEx_Calibration_Start(&hadc2,ADC_SINGLE_ENDED); //  NTC Temp, PCB Temp, Input Voltage
		HAL_ADCEx_Calibration_Start(&hadc5,ADC_SINGLE_ENDED); // IC Temp

		//HAL_ADC_Start_DMA(&hadc1, (uint32_t*)dataAdc1, SAMPLEWINDOW1);
		HAL_ADC_Start_DMA(&hadc2, (uint32_t*)dataAdc2, 3); // NTC Temp, PCB Temp, Input Voltage
		HAL_ADC_Start_DMA(&hadc5, (uint32_t*)dataAdc5, 2); // IC Temp
}


float Measurement_GetInputVoltage()
{
	return ((((float)dataAdc2[2] * (float)VREF) / ADCOVERSAMPLERESOLTION) / INPUTVOLTAGE_RESISTOR2) * (INPUTVOLTAGE_RESISTOR1 + INPUTVOLTAGE_RESISTOR2);
}

float Measurement_GetTemperatureNtc()
{
	float adcVal = dataAdc2[0];
	if(adcVal == 0.0f)
	{
	return 0;
	}
	float rntc = (float)NTC_RESISTOR1 / (((float)ADCOVERSAMPLERESOLTION / (float)adcVal ) - 1.0f);
	float temp = 0;
	if((float)*get_ntcBeta() != 0)
	{
		temp = rntc / (float)NTC_NOMINAL_R;
		temp = logf(temp);
		temp /= (float)*get_ntcBeta();
		temp += 1.0f / ((float)NTC_NOMINAL_TEMP + 273.15f);
		if(temp == 0)
		{
			return 0;
		}
		temp = 1.0f / temp;
		temp -= 273.15f;
	}
	return temp;
}

float Measurement_GetPcbTemperature()
{
	float adcVal = dataAdc2[1];
	return ((((float)adcVal * (float)VREF) / ADCOVERSAMPLERESOLTION) - PCB_TEMP_SENSOR_V0) / PCB_TEMP_SENSOR_TC;
}



#define INTERNALTEMP_AVGSLOPE	2.5f // mV/°C
#define AVG_SLOPE (2.5F)
#define V_AT_25C  (1.43F)
#define V_REF_INT (1.2F)

float Measurement_GetIcTemperature()
{
	float adcValSense = dataAdc5[0];
	//float adcValRef = dataAdc5[1];
	uint16_t calibrationData30 = *((uint16_t*)0x1FFF75A8);	// Calibration adc value at 30 °C and 3V
	uint16_t calibrationData130 = *((uint16_t*)0x1FFF75CA);
	float temp = (((float)(130 - 30) / (float)(calibrationData130 - calibrationData30)) * (adcValSense *  ((float)VREF/(float)3.0) - calibrationData30)) + 30;

	//float vRef = (((float)adcValRef * (float)VREF) / ADCRESOLUTION);
	//float vSense = (((float)adcValSense * (float)VREF) / ADCRESOLUTION);

	//V_Sense = (float)(AD_RES[1] * V_Ref) / 4095.0;
	//Temperature = (((V_AT_25C - V_Sense) * 1000.0) /AVG_SLOPE) + 25.0;
	return temp;
}



/*==============================================================================*/
/* Private functions                                                          	*/
/*==============================================================================*/

/*==============================================================================*/
/* Callback functions                                                          	*/
/*==============================================================================*/


