/**
  ******************************************************************************
  * @file           : dmx_512.c
  * @brief          : Handle DMX and RDM phyiscal interface
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Kuenzler Technologies GbR
  * All rights reserved.
  *
  *
  ******************************************************************************
*/

/*==============================================================================*/
/* Includes                                                                  	*/
/*==============================================================================*/
#include "dmx_512.h"
#include "variables.h"
#include "math.h"
#include "string.h"

/*==============================================================================*/
/* Defines & Macros                                                           	*/
/*==============================================================================*/
#define DMXBUFFERLENGTH 600
#define DMXLENGTH		512

#define DMX_STARTCODE	0
#define RDM_STARTCODE	0xCC

typedef enum
{
	DMX_IDLE,
	DMX_SEARCH_BREAK,
	DMX_RECEIVE_DATA,
	DMX_RECEIVED_DATA_RTO,
	DMX_RECEIVED_DATA,
	DMX_SEND_BREAK,
	DMX_SENT_BREAK,
	DMX_SEND_DATA,
} dmx_state_t;
/*==============================================================================*/
/* Private Variables                                                         	*/
/*==============================================================================*/
static uint8_t rxBufferDMX[DMXBUFFERLENGTH];
static uint8_t txFeBufferDmx[DMXBUFFERLENGTH];
static uint16_t txLength = 0;
static dmx_state_t state = DMX_IDLE;

static UART_HandleTypeDef* huartDmx;
static GPIO_TypeDef* dirPort;
static uint16_t dirPin;

static void (*dmx_receive_handler)(const uint8_t * data, uint16_t length);
static void (*rdm_receive_handler)(const uint8_t * data, uint16_t length);
/*==============================================================================*/
/* Private Function declarations                                                     	*/
/*==============================================================================*/
static void StartReception(bool timeout);
static void ReceiveData();
static void ChangeBaudrate(uint32_t baudrate);
/*==============================================================================*/
/* Function definitions                                                     	*/
/*==============================================================================*/
void InitDMX512(UART_HandleTypeDef* _huartDmx,GPIO_TypeDef* _dirPort, uint16_t _dirPin, void (*dmxCallback)(const uint8_t * msg, uint16_t length), void (*rdmCallback)(const uint8_t * msg, uint16_t length))
{
	huartDmx = _huartDmx;
	dirPort = _dirPort;
	dirPin = _dirPin;
	txLength = 0;
	dmx_receive_handler = dmxCallback;
	rdm_receive_handler = rdmCallback;
	HAL_UART_ReceiverTimeout_Config(huartDmx, 100);
	state = DMX_IDLE;
}

void DMX512_Task()
{
	switch(state)
	{
		case DMX_IDLE:
			// Restart UART and search for break
			if(huartDmx->RxState == HAL_UART_STATE_READY)
			{
				HAL_GPIO_WritePin(dirPort, dirPin, false);
				state = DMX_SEARCH_BREAK;
				// clear RxDatastream
				memset(rxBufferDMX,0,sizeof(rxBufferDMX));
				ChangeBaudrate(250000);
				StartReception(false);
			}
		break;
		case DMX_SEARCH_BREAK:
			// Do noting and wait for Error Hander FE
		break;
		case DMX_RECEIVE_DATA:
			// Do noting and wait for Error Hander RTO Flag
		break;
		case DMX_RECEIVED_DATA_RTO:
			state = DMX_IDLE;
			ReceiveData();
		break;
		case DMX_RECEIVED_DATA:
			// Should not occur
		break;
		case DMX_SEND_BREAK:
			// Do noting and wait for Transmit complete
		break;
		case DMX_SENT_BREAK:
			// Send Data
			state = DMX_SEND_DATA;
			ChangeBaudrate(250000);
			if(HAL_UART_Transmit_IT(huartDmx,&txFeBufferDmx[1],txLength) != HAL_OK)
			{
				HAL_UART_Abort_IT(huartDmx);
				state = DMX_IDLE;
			}
			break;
		case DMX_SEND_DATA:
			// Do noting and wait for Transmit complete
		break;
		default:
		break;
	}
}

void DMX512_WriteRdmFrame(const uint8_t* data, uint16_t length)
{
	if(length < (DMXBUFFERLENGTH - 1))
	{
		// Abort Reception
		HAL_GPIO_WritePin(dirPort, dirPin, true);
		memcpy(&txFeBufferDmx[1],data,length);
		txLength = length;

		ChangeBaudrate(38400);
		txFeBufferDmx[0] = 0;
		if(HAL_UART_Transmit_IT(huartDmx,txFeBufferDmx,1) == HAL_OK)
		{
			state = DMX_SEND_BREAK;
		}
		else
		{
			HAL_UART_Abort_IT(huartDmx);
			state = DMX_IDLE;
		}
	}
}

void DMX512_WriteData(const uint8_t* data, uint16_t length)
{
	if(length < (DMXBUFFERLENGTH - 1))
	{
		// Abort Reception
		HAL_GPIO_WritePin(dirPort, dirPin, true);
		memcpy(&txFeBufferDmx[1],data,length);
		txLength = length;
		state = DMX_SENT_BREAK;
	}
}

/*==============================================================================*/
/* Private Function definitions                                                	*/
/*==============================================================================*/
static void StartReception(bool dataStream)
{
	HAL_StatusTypeDef ret;
	if(dataStream)
	{
		HAL_UART_EnableReceiverTimeout(huartDmx);
		ret = HAL_UART_Receive_DMA(huartDmx, rxBufferDMX, DMXBUFFERLENGTH);
	}
	else
	{
		HAL_UART_DisableReceiverTimeout(huartDmx);
		ret = HAL_UART_Receive_DMA(huartDmx, txFeBufferDmx, DMXBUFFERLENGTH);
	}
	if(ret != HAL_OK)
	{
		HAL_UART_Abort_IT(huartDmx);
		state = DMX_IDLE;
	}
	//HAL_UARTEx_ReceiveToIdle_DMA(huart, pData, Size)
	//HAL_UART_RECEPTION_TORTO
}

static void ReceiveData()
{
	switch(rxBufferDMX[0])
	{
		case DMX_STARTCODE:
			if(dmx_receive_handler != NULL)
			{
				// Excluding StartCode
				dmx_receive_handler(&rxBufferDMX[1],DMXLENGTH);
			}
			break;
		case RDM_STARTCODE:
			if(rdm_receive_handler != NULL)
			{
				rdm_receive_handler(&rxBufferDMX[0],rxBufferDMX[2] + 2); // Message Length excluding CRC
			}
			break;
		default:
			break;
	}
}

static void ChangeBaudrate(uint32_t baudrate)
{
	if(huartDmx->Init.BaudRate != baudrate)
	{
		HAL_UART_Abort(huartDmx);
		HAL_UART_DeInit(huartDmx);
		huartDmx->Init.BaudRate = baudrate;
		HAL_UART_Init(huartDmx);
	}
}
/*==============================================================================*/
/* Callback functions                                                          	*/
/*==============================================================================*/

void DMX512_TxCpltCallback(UART_HandleTypeDef *huart)
{
	if(huart == huartDmx)
	{
		if(state == DMX_SEND_BREAK)
		{
			state = DMX_SENT_BREAK;
		}
		else if(state == DMX_SEND_DATA)
		{
			state = DMX_IDLE;
		}
	}
}

void HAL_UART_RxCpltCallback(UART_HandleTypeDef *huart)
{
	if(huart == huartDmx)
	{
		// Should not occur
		if(state == DMX_RECEIVED_DATA)
		{
			state = DMX_IDLE;
		}
		else if(state == DMX_RECEIVED_DATA_RTO)
		{
			state = DMX_IDLE;
		}
		else if(state == DMX_SEND_BREAK || state == DMX_SENT_BREAK || state == DMX_SEND_DATA)
		{
			state = DMX_IDLE;
		}
		else
		{
			state = DMX_IDLE;
		}
	}
}

void HAL_UARTEx_RxEventCallback(UART_HandleTypeDef *huart, uint16_t Size)
{
	if(huart == huartDmx)
	{
		// Should not occur
		state = DMX_IDLE;
	}
	(void)Size;
}

void DMX512_ErrorCallback(UART_HandleTypeDef *huart)
{
	if(huart == huartDmx)
	{
		if((huart->ErrorCode & HAL_UART_ERROR_FE) > 0)
		{
			if(state == DMX_SEARCH_BREAK)
			{
				// Start receiving again with timeout. Do it here to make sure we are receiving data fast enough
				state = DMX_RECEIVE_DATA;
				StartReception(true);
			}
			else
			{
				// Abort Operation
				state = DMX_IDLE;
			}

		}
		else if((huart->ErrorCode & HAL_UART_ERROR_RTO) > 0 )
		{
			if(state == DMX_RECEIVE_DATA)
			{
				if((DMXBUFFERLENGTH - huart->hdmarx->Instance->CNDTR) > 0)
				{
					state = DMX_RECEIVED_DATA_RTO;
				}
				else
				{
					state = DMX_IDLE;
				}
			}
			else
			{
				// Abort Operation
				state = DMX_IDLE;
			}
		}
		else
		{
			state = DMX_IDLE;
		}
	}
}
