/**
  ******************************************************************************
  * @file           : counter.c
  * @brief          : Simple implementation for rdm counter
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Kuenzler Technologies GbR
  * All rights reserved.
  *
  *
  ******************************************************************************
*/

/*==============================================================================*/
/* Includes                                                                  	*/
/*==============================================================================*/

#include "counter.h"
#include "variables.h"
#include "runtime_implementation.h"

/*==============================================================================*/
/* Defines & Macros                                                           	*/
/*==============================================================================*/

/*==============================================================================*/
/* Private Variables                                                         	*/
/*==============================================================================*/
static bool deviceMinuteTimeSet;
static uint32_t deviceMinuteLastTime;

static uint32_t countSourceMs;
static bool sourceMsTimeSet;
static uint32_t sourceMsLastTime;
/*==============================================================================*/
/* Private function declarations                                               	*/
/*==============================================================================*/

static void CountDeviceMinute(uint32_t tickMs);
static void CountSourceMinute(uint32_t tickMs);
/*==============================================================================*/
/* Function definitions                                                     	*/
/*==============================================================================*/
void InitCounter()
{
	deviceMinuteTimeSet = false;
	deviceMinuteLastTime = 0;
	sourceMsTimeSet = false;
	countSourceMs = 0;
	sourceMsLastTime = 0;
}


void CounterTask(uint32_t tickMs)
{
	CountDeviceMinute(tickMs);
	CountSourceMinute(tickMs);
}

void IncrementPowercycle()
{
	/* update powercycle counter - 更新电源循环计数器*/
	set_powerCycles((*get_powerCycles() + 1));
}

/*==============================================================================*/
/* Private function definitions                                                	*/
/*==============================================================================*/

static void CountDeviceMinute(uint32_t tickMs)
{
	if(!deviceMinuteTimeSet)
	{
		deviceMinuteLastTime = tickMs;
		deviceMinuteTimeSet = true;
	}
	else if(tickMs >= deviceMinuteLastTime + MINUTES_MS)
	{
		set_deviceMinutes(*get_deviceMinutes() +1);
		deviceMinuteLastTime = tickMs;
	}
	else if(tickMs < deviceMinuteLastTime)
	{
		deviceMinuteLastTime = tickMs;
	}
}

static void CountSourceMinute(uint32_t tickMs)
{
	if(RuntimeImpLightisOn())
	{
		if(sourceMsTimeSet)
		{
			if(tickMs != sourceMsLastTime)
			{
				countSourceMs++;
				sourceMsLastTime = tickMs;
				if(countSourceMs >= (60 *1000))
				{
					countSourceMs = 0;
					set_sourceMinutes(*get_sourceMinutes() +1);
				}
			}
		}
		else
		{
			sourceMsTimeSet = true;
			sourceMsLastTime = tickMs;
		}
	}
	else
	{
		sourceMsTimeSet = false;
	}
}
